package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid;

import java.util.Collection;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.problem.ProblemDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.fmdomain.FMProblem;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.ProblemRepository;

public class ProblemDataLayerOvid implements ProblemDataLayer {
	private static final String INACTIVE_STATUS = "inactive";

	private static final Log logger = LogFactory.getLog(ProblemDataLayerOvid.class);

	final private Date defaultBeginDate = DateHelper.parseDate("01/01/1990");
	final private Date defaultEndDate = new Date();

	public Problems fetchProblems(String patientId) {
		Problems allProblems = getPatientProblems(patientId);
		Problems activeProblems = new Problems();
		for (Problem problem : allProblems) {
			if (problem.isActive()) {
				activeProblems.add(problem);
			}
		}
		return activeProblems;
	}

	private Problems getPatientProblems(String patientId) {
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		ProblemRepository problemRepo = connectionManager.getProblemRepository();

		Collection<FMProblem> ovidProblems;
		try {
			logger.info(String.format("retrieve problems [patient=%s]", patientId));
			ovidProblems = problemRepo.getProblemsForDFN(patientId, defaultBeginDate, defaultEndDate);
			logger.info(String.format("retrieved %s problems [patient=%s]", ovidProblems.size(), patientId));
		} catch (OvidDomainException e) {
			throw new RuntimeException("Unable to retrieve problem data from OVID", e);
		} finally {
			connectionManager.releaseRepositoryConnections(problemRepo);
		}

		Problems problems = transformFromOvidProblems(patientId, ovidProblems);
		return problems;
	}

	private Problems transformFromOvidProblems(String patientId, Collection<FMProblem> ovidProblems) {
		Problems problems = new Problems();
		for (FMProblem ovidProblem : ovidProblems) {
			logger.debug("toString: " + ovidProblem.toString());

			Problem p = new Problem();
			p.setDescription(ovidProblem.getIcdDiagnosis().getDiagnosis());
			p.setOnsetDate(ovidProblem.getDateOfOnset());
			p.setPatientId(patientId);
			p.setResolutionDate(ovidProblem.getDateResolved());
			p.setProblemId(ovidProblem.getIEN());
			p.setPriority(ovidProblem.getPriority());
			p.setActive(calculateIsActive(ovidProblem.getStatus()));
			problems.add(p);
		}
		return problems;
	}

	private boolean calculateIsActive(String status) {
		boolean isActive;
		if (NullChecker.isNullish(status)) {
			isActive = true;
		} else if (status.toLowerCase().contentEquals(INACTIVE_STATUS)) {
			isActive = false;
		} else {
			isActive = true;
		}
		return isActive;
	}

	public Problems fetchProblems(String patientId, DateFilter dateFilter) {
		Problems problems = fetchProblems(patientId);
		
		return DateFilterer.filterByDate(problems, dateFilter);
	}

	@Override
	public Problems fetchProblems(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		
		return fetchProblems(patientIdentifier.getUniqueId(), dateFilter);
	}

}
