package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid;

import java.util.Collection;
import java.util.HashSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchCriteria;
import com.agilex.healthcare.mobilehealthplatform.domain.Address;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.NextOfKin;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.fmdomain.FMPatient;
import com.medsphere.fmdomain.FMPatientContact;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.PatientRepository;

public class PatientDataLayerOvidLegacy implements PatientDataLayer {
	private static final Log logger = LogFactory.getLog(PatientDataLayerOvid.class);

	@Override
	public Patients patientSearch(PatientSearchCriteria criteria) {
		logger.debug("performing patient search");
		Patients patients = new Patients();
		String searchString = determineSearchString(criteria);
		Collection<FMPatient> vistaPatients = lookupPatientInVista(searchString);

		if (NullChecker.isNotNullish(criteria.getSearchString()) && criteria.getSearchString().contentEquals(searchString) && NullChecker.isNullish(vistaPatients)) {
			criteria.setSearchString(null);
			searchString = determineSearchString(criteria);
			if (NullChecker.isNotNullish(searchString)) {
				logger.debug("Search by search string did not find any patients, attempting to search by discrete elements");
				vistaPatients = lookupPatientInVista(searchString);
			}
		}

		for (FMPatient ovidPatient : vistaPatients) {
			Patient patient = translateFromOvidPatient(ovidPatient);
			patients.add(patient);
		}

		logger.debug("complete with patient search, returning " + patients.size() + " record(s)");
		return patients;
	}

	private String determineSearchString(PatientSearchCriteria criteria) {
		String searchString = null;

		if (NullChecker.isNotNullish(criteria.getSearchString()))
			searchString = criteria.getSearchString();
		else if (criteria.getPrototype() != null && NullChecker.isNotNullish(criteria.getPrototype().getLastName())) {
			searchString = criteria.getPrototype().getLastName();
			if (criteria.getPrototype() != null && NullChecker.isNotNullish(criteria.getPrototype().getFirstName()))
				searchString = searchString + "," + criteria.getPrototype().getFirstName();
		} else if (criteria.getPrototype() != null && NullChecker.isNotNullish(criteria.getPrototype().getSsn()))
			searchString = criteria.getPrototype().getSsn().replace("-", "");
		else if (NullChecker.isNotNullish(criteria.getLastInitialPlusLastFourOfSsn()))
			searchString = criteria.getLastInitialPlusLastFourOfSsn();
		else if (NullChecker.isNotNullish(criteria.getLastFourOfSsn()))
			searchString = criteria.getLastFourOfSsn();

		return searchString;
	}

	private Patient translateFromOvidPatient(FMPatient vistaPatient) {
		Patient patient = new Patient();
		parseNameIntoPatient(patient, vistaPatient.getName());
		if (NullChecker.isNotNullish(patient.getDisplayName()) || NullChecker.isNotNullish(patient.getLastName())) {
			patient.setId(vistaPatient.getIEN());
			patient.setSsn(vistaPatient.getSsn());
			patient.setGender(vistaPatient.getSex());
			patient.setDateOfBirth(new DOBDate(vistaPatient.getDob()));
			patient.setAge(vistaPatient.getDisplayAge());
			patient.setGender(vistaPatient.getSex());
		} else {
			patient = null;
		}
		return patient;
	}

	private Collection<FMPatient> lookupPatientInVista(String searchString) {
		if (NullChecker.isNullish(searchString))
			throw new IllegalArgumentException("Insufficient search criteria");

		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientRepository patientRepo = connectionManager.getPatientRepository();

		Collection<FMPatient> patients;
		try {
			logger.debug("performing search against patient repo, searchstring=" + searchString);
			patients = patientRepo.lookupPatientsByName(searchString);
			logger.debug("patient search complete");
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing lookup patients in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		return patients;
	}

	// private void debugPrint(Collection<FMPatient> patients) {
	// for (FMPatient patient : patients) {
	// debugPrint(patient);
	// }
	// }
	//
	// private void debugPrint(FMPatient patient) {
	// logger.debug("patient");
	// logger.debug("name: " + patient.getName());
	// logger.debug("family name: " + patient.getFamilyName());
	// logger.debug("given name: " + patient.getGivenName());
	// logger.debug("middle name: " + patient.getMiddleName());
	// logger.debug("dob: " + patient.getDob());
	// logger.debug("display age: " + patient.getDisplayAge());
	// logger.debug("age: " + patient.getAge());
	// logger.debug("ssn: " + patient.getSsn());
	// logger.debug("gender: " + patient.getSex());
	// logger.debug("id: " + patient.getId());
	// logger.debug("eid: " + patient.getEnterprisePatientIdentifier());
	// logger.debug("ien: " + patient.getIEN());
	// logger.debug("toString: " + patient.toString());
	// }

	@Override
	public PatientDemographics getDemographics(String id) {
		logger.debug("begin get demographics [patientId='" + id + "']");

		PatientDemographics demographics = new PatientDemographics();

		Collection<String> iens = new HashSet<String>();
		iens.add(id);

		Collection<FMPatientContact> ovidContacts;
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientRepository patientRepo = connectionManager.getPatientRepository();
		try {
			logger.debug("performing get contacts against patient repo, id=" + id);
			ovidContacts = patientRepo.getContacts(iens);
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get contacts in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		demographics = transformOvidContactToDemograhics(ovidContacts);

		logger.debug("complete get demographics");
		return demographics;
	}

	private PatientDemographics transformOvidContactToDemograhics(Collection<FMPatientContact> ovidContacts) {
		logger.debug("transforming ovid contact information to a demographics object");
		PatientDemographics demographics = new PatientDemographics();
		for (FMPatientContact ovidcontact : ovidContacts) {
			NextOfKin nextOfKin = new NextOfKin();
			nextOfKin.setName(ovidcontact.getNokNameOfPrimary());
			nextOfKin.setCity(ovidcontact.getNokCity());
			// nextOfKin.setState(ovidcontact.getNokState());
			nextOfKin.setStreetAddressLine1(ovidcontact.getNokStreetAddressLine1());
			nextOfKin.setStreetAddressLine2(ovidcontact.getNokStreetAddressLine2());
			nextOfKin.setStreetAddressLine3(ovidcontact.getNokStreetAddressLine3());
			nextOfKin.setRelationship(ovidcontact.getNokRelationshipToPatient());
			nextOfKin.setPhoneNumber(ovidcontact.getNokPhoneNumber());
			nextOfKin.setPhoneNumberWork(ovidcontact.getNokWorkPhoneNumber());
			demographics.setNextOfKin(nextOfKin);

			Address address = new Address();
			address.setStreetAddressLine1(ovidcontact.getStreetAddressLine1());
			address.setStreetAddressLine2(ovidcontact.getStreetAddressLine2());
			address.setStreetAddressLine3(ovidcontact.getStreetAddressLine3());
			address.setCity(ovidcontact.getCity());
			address.setState(ovidcontact.getStateValue());
			address.setZipCode(ovidcontact.getZipCode());
			demographics.setAddress(address);

			demographics.setEmailAddress(ovidcontact.getEmailAddress());

			demographics.setPhoneNumberPager(ovidcontact.getPagerNumber());
			demographics.setPhoneNumberMobile(ovidcontact.getPhoneNumberCellular());
			demographics.setPhoneNumberHome(ovidcontact.getPhoneNumberResidence());
			demographics.setPhoneNumberWork(ovidcontact.getPhoneNumberWork());
		}
		return demographics;
	}

	@Override
	public Patient getPatient(String id) {
		FMPatient ovidPatient = getOvidPatient(id);
		Patient patient = translateFromOvidPatient(ovidPatient);
		return patient;
	}

	private FMPatient getOvidPatient(String id) {
		Collection<String> iens = new HashSet<String>();
		iens.add(id);

		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientRepository patientRepo = connectionManager.getPatientRepository();
		Collection<FMPatient> ovidPatients;
		try {
			logger.debug("performing get contacts against patient repo, id=" + id);
			ovidPatients = patientRepo.getPatientsForIENS(iens);
			logger.debug("complete getting contacts against patient repo, id=" + id);
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get contacts in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		// debugPrint(ovidPatients);

		FMPatient patient = null;
		for (FMPatient ovidPatient : ovidPatients) {
			patient = ovidPatient;
			break;
		}
		return patient;
	}

	// TODO: i can do better than this!
	private void parseNameIntoPatient(Patient patient, String unparsedName) {
		patient.setDisplayName(unparsedName);
		if (NullChecker.isNotNullish(unparsedName)) {
			String[] parsedNameParts = unparsedName.split(",");
			if (parsedNameParts.length >= 1) {
				patient.setLastName(parsedNameParts[0]);
			}
			if (parsedNameParts.length >= 2) {
				patient.setFirstName(parsedNameParts[1]);
			}
		}
	}

}
