package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.MHPConstants;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchCriteria;
import com.agilex.healthcare.mobilehealthplatform.domain.Address;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.NextOfKin;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.ovid.OvidPatientRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidPatient;
import com.agilex.healthcare.mobilehealthplatform.ovid.util.OvidDateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.medsphere.fmdomain.FMPatient;
import com.medsphere.fmdomain.FMPatientContact;
import com.medsphere.fmdomain.FMPatientMovement;
import com.medsphere.ovid.domain.ov.OvidDomainException;
import com.medsphere.ovid.domain.ov.PatientMovementRepository;
import com.medsphere.ovid.domain.ov.PatientRepository;

public class PatientDataLayerOvid implements PatientDataLayer {
	private static final Log logger = LogFactory.getLog(PatientDataLayerOvid.class);

	@Override
	public Patients patientSearch(PatientSearchCriteria criteria) {
		logger.debug("performing patient search");
		Patients patients = new Patients();
		String searchString = criteria.determineSearchString();
		Collection<OvidPatient> vistaPatients = lookupPatientInVista(searchString);

		for (OvidPatient ovidPatient : vistaPatients) {
			Patient patient = translateFromOvidPatient(ovidPatient);
			patients.add(patient);
		}

		logger.debug(String.format("complete with patient search [searchstring=%s], returning %s record(s)", searchString, patients.size()));
		return patients;
	}


	private Patient translateFromOvidPatient(OvidPatient vistaPatient) {
		Patient patient = new Patient();
		parseNameIntoPatient(patient, vistaPatient.getName());
		if (NullChecker.isNotNullish(patient.getDisplayName()) || NullChecker.isNotNullish(patient.getLastName())) {
			patient.setId(vistaPatient.getIen());
			patient.setSsn(vistaPatient.getSsn());
			patient.setGender(vistaPatient.getSex());
			patient.setAge(vistaPatient.getDisplayAge());
			patient.setDateOfBirth(OvidDateHelper.createDOBDate(vistaPatient.getDob()));
			patient.setWardLocation(vistaPatient.getWardLocation());
			patient.setRoombed(vistaPatient.getRoomBed());

			if (NullChecker.isNotNullish(vistaPatient.getStatus()) && MHPConstants.INPATIENT.equals(vistaPatient.getStatus())) {
				patient.setInpatient(true);
			}

		} else {
			patient = null;
		}
		return patient;
	}

	private Patient translateFromOvidPatient(FMPatient vistaPatient) {
		Patient patient = new Patient();	
		parseNameIntoPatient(patient, vistaPatient.getName());
		if (NullChecker.isNotNullish(patient.getDisplayName()) || NullChecker.isNotNullish(patient.getLastName())) {
			patient.setId(vistaPatient.getIEN());
			patient.setSsn(vistaPatient.getSsn());
			patient.setGender(vistaPatient.getSex());
			
			String fmDob = vistaPatient.getValue("DATE OF BIRTH").trim();
			logger.debug("fmDob=" + fmDob);
			if (NullChecker.isNotNullish(fmDob)) {
				// workaround for date of birth sue to an issue where FMPatient is returning bogus ("0101" or "01") for
				// date and month when none was entered.							
				if (fmDob.endsWith("0101") || fmDob.endsWith("01")) {
					String dateOfBirth = getDateOfBirth(vistaPatient.getIEN());
					if (!dateOfBirth.equals(fmDob)) {
						fmDob = dateOfBirth;
					}
				}
				
				String dobStr = OvidDateHelper.fmDateToDobString(fmDob);
				logger.debug("++++++++++++++++++++  dobStr=" + dobStr);
				DOBDate dob = OvidDateHelper.createDOBDate(dobStr);
				patient.setDateOfBirth(dob);						
			}
			int age = vistaPatient.getAge().intValue();
			patient.setAge(String.valueOf(age));
			patient.setGender(vistaPatient.getSex());
			patient.setEnterpriseId(vistaPatient.getId());
			patient.setWardLocation(vistaPatient.getWardLocation());
			patient.setRoombed(vistaPatient.getRoomBed());
		} else {
			patient = null;
		}
		return patient;
	}

	private Collection<OvidPatient> lookupPatientInVista(String searchString) {
		if (NullChecker.isNullish(searchString))
			throw new IllegalArgumentException("Insufficient search criteria");

		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		OvidPatientRepository patientRepo = connectionManager.getOvidPatientRepository();

		Collection<OvidPatient> patients;
		try {
			logger.info(String.format("retrieve patients [searchstring=%s]", searchString));
			patients = patientRepo.lookupPatientsByName(searchString);
			logger.info(String.format("retrieved %s patients [searchstring=%s]", patients.size(), searchString));
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing lookup patients in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		return patients;
	}
	
	private String getDateOfBirth(String id) {
		String dob = null;
		
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		OvidPatientRepository patientRepo = connectionManager.getOvidPatientRepository();
		try {
			logger.info(String.format("retrieve patient date of birth [patientid=%s]", id));
			dob = patientRepo.getDateOfBirth(id);
			logger.info(String.format("retrieved patient date of birth [patientid=%s]", id));
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get patient date of birth in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}
		return dob;
	}
	
	@Override
	public PatientDemographics getDemographics(String id) {
		logger.debug("begin get demographics [patientId='" + id + "']");

		PatientDemographics demographics = new PatientDemographics();

		Collection<String> iens = new HashSet<String>();
		iens.add(id);

		Collection<FMPatientContact> ovidContacts;
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientRepository patientRepo = connectionManager.getPatientRepository();
		try {
			logger.info(String.format("retrieve patient [patientid=%s]", id));
			ovidContacts = patientRepo.getContacts(iens);
			logger.info(String.format("retrieved patient [patientid=%s]", id));
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get contacts in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		demographics = transformOvidContactToDemograhics(ovidContacts);

		logger.debug("complete get demographics");
		return demographics;
	}

	private PatientDemographics transformOvidContactToDemograhics(Collection<FMPatientContact> ovidContacts) {
		logger.debug("transforming ovid contact information to a demographics object");
		PatientDemographics demographics = new PatientDemographics();
		for (FMPatientContact ovidcontact : ovidContacts) {
			NextOfKin nextOfKin = new NextOfKin();
			nextOfKin.setName(ovidcontact.getNokNameOfPrimary());
			nextOfKin.setCity(ovidcontact.getNokCity());
			// nextOfKin.setState(ovidcontact.getNokState());
			nextOfKin.setStreetAddressLine1(ovidcontact.getNokStreetAddressLine1());
			nextOfKin.setStreetAddressLine2(ovidcontact.getNokStreetAddressLine2());
			nextOfKin.setStreetAddressLine3(ovidcontact.getNokStreetAddressLine3());
			nextOfKin.setRelationship(ovidcontact.getNokRelationshipToPatient());
			nextOfKin.setPhoneNumber(ovidcontact.getNokPhoneNumber());
			nextOfKin.setPhoneNumberWork(ovidcontact.getNokWorkPhoneNumber());
			demographics.setNextOfKin(nextOfKin);

			Address address = new Address();
			address.setStreetAddressLine1(ovidcontact.getStreetAddressLine1());
			address.setStreetAddressLine2(ovidcontact.getStreetAddressLine2());
			address.setStreetAddressLine3(ovidcontact.getStreetAddressLine3());
			address.setCity(ovidcontact.getCity());
			address.setState(ovidcontact.getStateValue());
			address.setZipCode(ovidcontact.getZipCode());
			demographics.setAddress(address);

			demographics.setEmailAddress(ovidcontact.getEmailAddress());

			demographics.setPhoneNumberPager(ovidcontact.getPagerNumber());
			demographics.setPhoneNumberMobile(ovidcontact.getPhoneNumberCellular());
			demographics.setPhoneNumberHome(ovidcontact.getPhoneNumberResidence());
			demographics.setPhoneNumberWork(ovidcontact.getPhoneNumberWork());
		}
		return demographics;
	}

	@Override
	public Patient getPatient(String id) {		
		FMPatient ovidPatient = getFMPatient(id);
		Patient patient = translateFromOvidPatient(ovidPatient);
		try {
			String status = getPatientStatus(ovidPatient.getIEN());
			if (NullChecker.isNotNullish(status) && MHPConstants.INPATIENT.equals(status)) {
				patient.setInpatient(true);
			}
		} catch (OvidDomainException oe) {
			logger.error("Error occurred while performing get patient status", oe);
		}
						
		return patient;
	}

	private FMPatient getFMPatient(String id) {
		Collection<String> iens = new HashSet<String>();
		iens.add(id);

		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientRepository patientRepo = connectionManager.getPatientRepository();
		Collection<FMPatient> fmPatients;
		try {
			logger.debug("performing get contacts against patient repo, id=" + id);
			fmPatients = patientRepo.getPatientsForIENS(iens);
			logger.debug("complete getting contacts against patient repo, id=" + id);
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get contacts in repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(patientRepo);
		}

		// debugPrint(ovidPatients);
		
		return fmPatients.iterator().next();
	}
	
	// TODO: i can do better than this!
	private void parseNameIntoPatient(Patient patient, String unparsedName) {
		patient.setDisplayName(unparsedName);
		if (NullChecker.isNotNullish(unparsedName)) {
			String[] parsedNameParts = unparsedName.split(",");
			if (parsedNameParts.length >= 1) {
				patient.setLastName(parsedNameParts[0]);
			}
			if (parsedNameParts.length >= 2) {
				patient.setFirstName(parsedNameParts[1]);
			}
		}
	}

	private String getPatientStatus(String dfn) throws OvidDomainException {
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		PatientMovementRepository repo = connectionManager.getPatientMovementRepository();
		Collection<FMPatientMovement> list = new ArrayList<FMPatientMovement>();
		try {
			logger.debug("performing get status against patient movement repo, id=" + dfn);
			list = repo.getPatientMovementByPatientDFN(dfn);
			logger.debug("complete getting status against patient movement repo, id=" + dfn);
			
			if (list.size() == 0) {
				return MHPConstants.OUTPATIENT;
			}

			FMPatientMovement patientMovement = list.iterator().next();
			if (list.size() > 1) {
				//find the latest movement
				for (FMPatientMovement tempMovement : list) {
					if (patientMovement.getDateTime().getTime() < tempMovement.getDateTime().getTime()) {
						patientMovement = tempMovement;
					}
				}
			}
			if (patientMovement.getTransaction().equals("ADMISSION")) {	// ADMISSION = 1
				return MHPConstants.INPATIENT; //active
			} else {
				return MHPConstants.OUTPATIENT; //inactive
			}					
		} catch (OvidDomainException e) {
			throw new RuntimeException("Error occurred while performing get status in patient movement repo", e);
		} finally {
			connectionManager.releaseRepositoryConnections(repo);
		}
	}	
}
