package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid;

import java.util.Collection;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.ovid.MedicationRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidMedication;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.NullSafeStringComparer;
import com.medsphere.ovid.domain.ov.OvidDomainException;

public class MedicationDataLayerOvid implements MedicationDataLayer {
	private static final Log logger = LogFactory.getLog(MedicationDataLayerOvid.class);
	private Date currentDate = null;

	@Override
	public Medication getMedicationLite(PatientIdentifier patientIdentifier, String orderNumber) {
		String patientId = patientIdentifier.getUniqueId();
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		MedicationRepository repo = connectionManager.getMedicationRepository();

		OvidMedication ovidMedication = null;
		try {
			logger.info(String.format("retrieve medication (lite) [patient=%s][order=%s]", patientId, orderNumber));
			ovidMedication = repo.getMedicationLite(orderNumber);
			logger.info(String.format("retrieved medication (lite) [patient=%s][order=%s]", patientId, orderNumber));
		} catch (OvidDomainException e) {
			logger.error("Exception while retrieving medication for patient id: " + patientId + " order number:" + orderNumber, e);
			throw new RuntimeException(e);
		} finally {
			connectionManager.releaseRepositoryConnections(repo);
		}

		if (ovidMedication == null) {
			return null;
		}

		Medication medication = transformFromOvidMedication(patientId, ovidMedication);
		return medication;
	}

	@Override
	public Medication getMedicationDetail(PatientIdentifier patientIdentifier, String orderNumber) {
		String patientId = patientIdentifier.getUniqueId();
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		MedicationRepository repo = connectionManager.getMedicationRepository();

		OvidMedication ovidMedication = null;
		try {
			logger.info(String.format("retrieve medication (detail) [patient=%s][order=%s]", patientId, orderNumber));
			ovidMedication = repo.getOvidMedicationDetail(patientId, orderNumber);
			logger.info(String.format("retrieved medication (detail) [patient=%s][order=%s]", patientId, orderNumber));
		} catch (OvidDomainException e) {
			logger.error("Exception while retrieving medication for patient id: " + patientId + " order number:" + orderNumber, e);
			throw new RuntimeException(e);
		} finally {
			connectionManager.releaseRepositoryConnections(repo);
		}

		if (ovidMedication == null) {
			return null;
		}

		Medication medication = transformFromOvidMedication(patientId, ovidMedication);
		return medication;
	}

	@Override
	public Medications getMedications(PatientIdentifier patientIdentifier, boolean activeOnly) {
		String patientId = patientIdentifier.getUniqueId();
		logger.debug(String.format("begin getMedications [patientId=%s][activeOnly=%s]", patientId, Boolean.toString(activeOnly)));
		Medications medications = retrieveMedications(patientId);
		logger.debug(String.format("retrieved %s records.", medications.size()));
		if (activeOnly) {
			logger.debug("applying active only filter");
			medications = filterForActiveMedsOnly(medications);
			logger.info(String.format("active med filter applied, retrieved %s medications [patient=%s]", medications.size(), patientId));
		}
		logger.debug(String.format("returning %s records.", medications.size()));
		return medications;
	}

	private Medications filterForActiveMedsOnly(Medications medications) {
		Medications activeMedications = new Medications();
		for (Medication medication : medications) {
			if (medication.isActive()) {
				activeMedications.add(medication);
			}
		}
		return activeMedications;
	}

	private Medications retrieveMedications(String patientId) {
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		MedicationRepository repo = connectionManager.getMedicationRepository();

		Collection<OvidMedication> ovidMedications = null;
		try {
			logger.info(String.format("retrieve medications [patient=%s] ", patientId));
			ovidMedications = repo.getAllMedications(patientId);
			logger.info(String.format("retrieved %s medications [patient=%s]", ovidMedications.size(), patientId));
		} catch (OvidDomainException e) {
			logger.error("Exception while retrieving medications for patient id: " + patientId, e);
			throw new RuntimeException(e);
		} finally {
			connectionManager.releaseRepositoryConnections(repo);
		}

		Medications medications = transformFromOvidMedication(patientId, ovidMedications);
		return medications;
	}

	private Medication transformFromOvidMedication(String patientId, OvidMedication ovidMedication) {
		Medication medication = new Medication();
		medication.setDrugName(formatDrugName(ovidMedication.getMedication()));
		medication.setStartDate(ovidMedication.getStartDate());
		medication.setEndDate(ovidMedication.getStopDate());
		medication.setOrderNumber(ovidMedication.getOrderNumber());
		medication.setMedicationSource(ovidMedication.getMedType());
		medication.setSig(formatSig(ovidMedication.getSigDetail()));
		medication.setLastFilledDate(ovidMedication.getLastFilled());
		medication.setRefills(extractRefills(ovidMedication));
		medication.setQuantity(ovidMedication.getQuantity());
		medication.setDaysSupply(ovidMedication.getSupply());
		medication.setPrescriptionId(ovidMedication.getPrescriptionId());
		medication.setStatus(ovidMedication.getStatus());
		medication.setMedicationDetail(ovidMedication.getMedicationDetail());
		medication.setActive(calculateIsActive(ovidMedication));
		return medication;
	}

	private boolean calculateIsActive(OvidMedication ovidMedication) {
		MedicationActiveCalculator calculator = new MedicationActiveCalculator(getCurrentDate());
		boolean isActive = calculator.determineIsActive(ovidMedication);
		return isActive;
	}

	private Medications transformFromOvidMedication(String patientId, Collection<OvidMedication> ovidMedications) {
		Medications medications = new Medications();
		for (OvidMedication ovidMedication : ovidMedications) {
			Medication medication = transformFromOvidMedication(patientId, ovidMedication);
			medications.add(medication);
		}
		return medications;
	}

	private String extractRefills(OvidMedication ovidMedication) {
		return ovidMedication.getRefills();
	}

	private String formatSig(String sig) {
		if (NullChecker.isNullish(sig))
			return sig;
		return sig.replace("Sig:", "").trim();
	}

	private String formatDrugName(String s) {
		String ellipsis = "...";
		if (NullChecker.isNotNullish(s) && (s.length() > ellipsis.length()))
			s = s.replace(ellipsis, "");
		return s;
	}

	public void setCurrentDate(Date date) {
		this.currentDate = date;
	}

	private Date getCurrentDate() {
		if (NullChecker.isNotNullish(this.currentDate))
			return this.currentDate;
		else
			return new Date();
	}

	@Override
	public Medications getMedications(PatientIdentifier patientIdentifier, MedicationFilter medicationFilter) {
		String patientId = patientIdentifier.getUniqueId();
		Medications medications = retrieveMedications(patientId);
		if (isActive(medicationFilter)) {
			return filterForActiveMedsOnly(medications);
		}
		return medications;
	}

	public boolean isActive(MedicationFilter filter) {
		if (NullChecker.isNotNullish(filter.getStatuses())) {
			for (String status : filter.getStatuses()) {
				return NullSafeStringComparer.areEqual(status, "ACTIVE");
			}
		}
		return false;
	}
}
