package com.agilex.healthcare.mobilehealthplatform.ovid.repositories;

import static com.agilex.healthcare.testutility.AssertEx.assertEqualsIgnoreCase;
import static junit.framework.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Collection;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.MHPConstants;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.ovid.OvidPatientRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidPatient;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;
import com.medsphere.ovid.domain.ov.OvidDomainException;

public class OvidPatientRepositoryTest {

	private OvidPatient testPatient;
	private OvidPatientRepository repo;
	private IOvidConnectionManager connectionManager;

	@Before
	public void setUp() {
		PatientTestHelper.initializeOvidConnectionManager();
		testPatient = IntegrationTestConfiguration.getTestPatientForLookup();
		connectionManager = ConnectionManagerFactory.getInstance();
		repo = connectionManager.getOvidPatientRepository();
	}

	@After
	public void tearDown() {
		connectionManager.releaseRepositoryConnections(repo);
	}

	@Test
	public void validatePatientSearchReturnsValidData() throws Exception {
		Collection<OvidPatient> patients = repo.lookupPatientsByName(testPatient.getFamilyName());
		validatePatientData(testPatient, patients.iterator().next());
	}

	@Test
	public void validatePatientStatus() throws Exception {
		Collection<OvidPatient> patients = repo.lookupPatientsByName(testPatient.getFamilyName());
		assertEquals(patients.iterator().next().getStatus(), testPatient.getStatus());

		PatientTestHelper.initializeOvidConnectionManager();
		Patient inpatient = IntegrationTestConfiguration.getTestInpatient();		
		patients = repo.lookupPatientsByName(inpatient.getSsn());
		assertEquals(patients.iterator().next().getStatus(), MHPConstants.INPATIENT);
		
	}
	
	
	@Test
	public void validateInpatientWardLocation() throws Exception {
		PatientTestHelper.initializeOvidConnectionManager();
		Patient inpatient = IntegrationTestConfiguration.getTestInpatient();
		Collection<OvidPatient> patients = repo.lookupPatientsByName(inpatient.getSsn());
		OvidPatient ovidPatient = patients.iterator().next();
		assertEqualsIgnoreCase(inpatient.getWardLocation(), ovidPatient.getWardLocation());
		assertEquals(inpatient.getRoombed(), ovidPatient.getRoomBed());
	}
	
	private void validatePatientData(OvidPatient expectedPatient, OvidPatient retrievedPatient) {
		assertEqualsIgnoreCase(expectedPatient.getName(), retrievedPatient.getName());
		assertEquals(expectedPatient.getDob().toString(), retrievedPatient.getDob());		
	}

	@Test
	public void validatePatientSearchReturnsMultipleResults() throws OvidDomainException {
		OvidPatient expectedPatient = IntegrationTestConfiguration.getTestPatientForLookupWithMultipleResults();
		Collection<OvidPatient> patients = repo.lookupPatientsByName(expectedPatient.getFamilyName());
		assertTrue(patients.size() > 1);
	}

	@Test
	public void validatePatientContainsDOYearOnly() throws OvidDomainException {
		OvidPatient expectedPatient = IntegrationTestConfiguration.getTestPatientForLookupContainsDOBWithYearOnly();
		Collection<OvidPatient> patients = repo.lookupPatientsByName(expectedPatient.getSsn());
		validatePatientData(expectedPatient, patients.iterator().next());
	}

	@Test
	public void validatePatientContainsDOBMonthYearOnly() throws OvidDomainException {
		OvidPatient expectedPatient = IntegrationTestConfiguration
				.getTestPatientForLookupContainsDOBWithMonthYearOnly();
		Collection<OvidPatient> patients = repo.lookupPatientsByName(expectedPatient.getSsn());
		validatePatientData(expectedPatient, patients.iterator().next());
	}

}
