package com.agilex.healthcare.mobilehealthplatform.ovid.repositories;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.text.SimpleDateFormat;
import java.util.Collection;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.ConnectionManagerFactory;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.connectionmanagement.IOvidConnectionManager;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.ovid.MedicationRepository;
import com.agilex.healthcare.mobilehealthplatform.ovid.domain.OvidMedication;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;
import com.medsphere.ovid.domain.ov.OvidDomainException;

public class MedicationRepositoryTest {

	private MedicationRepository medicationRepo;
	private static final String BAD_PATIENT_ID = "Bad Patient Id";
	private static final String BAD_ORDER_ID = "Bad Order Id";
	private static final String BAD_PRESCRIPTION_ID = "Bad Prescription Id";

	@Before
	public void setUp() throws Exception {
		PatientTestHelper.initializeOvidConnectionManager();
		IOvidConnectionManager connectionManager = ConnectionManagerFactory.getInstance();
		medicationRepo = connectionManager.getMedicationRepository();
	}

	@After
	public void tearDown() throws Exception {
		ConnectionManagerFactory.getInstance().releaseRepositoryConnections(medicationRepo);
		medicationRepo = null;
	}

	@Ignore
	public void validateOutpatientMedication() throws OvidDomainException {
		validateMedication(IntegrationTestConfiguration.getOutpatientMedication());
	}

	@Test
	public void validateInpatientMedication() throws OvidDomainException {
		validateMedication(IntegrationTestConfiguration.getInpatientMedication2());
	}

	@Test
	public void validateOutpatientMedicationLite() throws OvidDomainException {
		validateMedicationLite(IntegrationTestConfiguration.getOutpatientMedication());
	}

	@Test
	public void validateInpatientMedicationLite() throws OvidDomainException {
		validateMedicationLite(IntegrationTestConfiguration.getInpatientMedication2());
	}

	private void validateMedication(Medication expectedMedication) throws OvidDomainException {
		OvidMedication retrievedMedication = findMedication(expectedMedication);
		assertNotNull(retrievedMedication);
		assertEquals(expectedMedication.getDrugName(), retrievedMedication.getMedication());
		assertEquals(expectedMedication.getEndDate(), retrievedMedication.getStopDate());
		assertEquals(expectedMedication.getStartDate(), retrievedMedication.getStartDate());
		assertEquals(expectedMedication.getStatus(), retrievedMedication.getStatus());
	}

	private OvidMedication findMedication(Medication expectedMedication) throws OvidDomainException {
		return findMedication(expectedMedication.getPatientId(), expectedMedication.getOrderNumber());
	}

	private OvidMedication findMedication(String patientId, String orderNumber) throws OvidDomainException {
		Collection<OvidMedication> medications = medicationRepo.getAllMedications(patientId);
		OvidMedication med = null;
		for (OvidMedication ovidMedication : medications) {
			if (ovidMedication.getOrderNumber().contentEquals(orderNumber)) {
				med = ovidMedication;
			}
		}
		return med;
	}

	private void validateMedicationLite(Medication expectedMedication) throws OvidDomainException {
		OvidMedication retrievedMedication = medicationRepo.getMedicationLite(expectedMedication.getOrderNumber());
		assertNotNull(retrievedMedication);
		System.err.println("retrievedMedication drug name=" + retrievedMedication.getMedication());
		System.err.println("expectedMedication drug name=" + expectedMedication.getDrugName());
		System.err.println("retrievedMedication status=" + retrievedMedication.getStatus());
		System.err.println("expectedMedication status=" + expectedMedication.getStatus());
		
		//check only the first 10 characters as the drug name is abbreviated using the lite version or may contain the generic notes
		assertEquals(expectedMedication.getDrugName().substring(0,10), retrievedMedication.getMedication().substring(0,10));
		Assert.assertNotSame(expectedMedication.getDrugName(), retrievedMedication.getMedication());		
		assertEquals(expectedMedication.getStatus(), retrievedMedication.getStatus());
	}

	// @Test
	public void validateGetMedicationHistoryWithBadPatientId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationHistory = medicationRepo.getMedicationHistory(BAD_PATIENT_ID, medication.getOrderNumber());
		assertTrue(medicationHistory == null);
	}

	// @Test
	public void validateGetMedicationHistoryWithBadOrderId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationHistory = medicationRepo.getMedicationHistory(medication.getPatientId(), BAD_ORDER_ID);
		System.out.println(medicationHistory);
		assertTrue(medicationHistory != null && medicationHistory.contains("MEDICATION HISTORY for"));
		assertTrue(medicationHistory.contains(IntegrationTestConfiguration
				.getMedicationHistoryStringForInpatientMedication()));
	}

	// @Test
	public void validateGetMedicationHistoryWithBadPatientIdAndBadOrderId() throws OvidDomainException {
		String medicationHistory = medicationRepo.getMedicationHistory(BAD_PATIENT_ID, BAD_ORDER_ID);
		assertTrue(medicationHistory == null);
	}

	// @Test
	public void validateGetMedicationDetailWithBadPatientId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationDetail = medicationRepo.getMedicationDetail(BAD_PATIENT_ID, medication.getPrescriptionId());
		assertTrue(medicationDetail == null);
	}

	// @Test
	public void validateGetMedicationDetailWithBadPrescriptionId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationDetail = medicationRepo.getMedicationDetail(medication.getPatientId(), BAD_PRESCRIPTION_ID);
		assertTrue(medicationDetail == null);
	}

	// @Test
	public void validateGetMedicationDetailWithBadOrderIdAndBadPrescriptionId() throws OvidDomainException {
		String medicationDetail = medicationRepo.getMedicationDetail(BAD_PATIENT_ID, BAD_PRESCRIPTION_ID);
		assertTrue(medicationDetail == null);
	}

//	@Test
	public void validateInpatientGetMedicationHistory() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationHistory = medicationRepo.getMedicationHistory(medication.getPatientId(),
				medication.getOrderNumber());
		System.out.println("Inpatient Medication History:\n" + medicationHistory);
		assertTrue(medicationHistory != null && medicationHistory.contains("MEDICATION HISTORY for"));
		assertTrue(medicationHistory.contains(IntegrationTestConfiguration
				.getMedicationHistoryStringForInpatientMedication()));
	}

//	@Test
	public void validateOutpatientGetMedicationHistory() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getOutpatientMedication();
		String medicationHistory = medicationRepo.getMedicationHistory(medication.getPatientId(),
				medication.getOrderNumber());
		System.out.println("Outpatient Medication History:\n" + medicationHistory);
		assertTrue(medicationHistory != null && medicationHistory.contains("MEDICATION HISTORY for"));
		assertTrue(medicationHistory.contains(IntegrationTestConfiguration
				.getMedicationHistoryStringForOutpatientMedication()));
	}

	@Test
	public void validateInpatientGetMedicationDetail() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		String medicationDetail = medicationRepo.getMedicationDetail(medication.getPatientId(),
				medication.getPrescriptionId());
		System.out.println("Inpatient Medication Detail:\n" + medicationDetail);
		assertTrue(medicationDetail != null && medicationDetail.contains("Medication:"));
		assertTrue(medicationDetail.contains(IntegrationTestConfiguration
				.getMedicationDetailStringForInpatientMedication()));
	}

	@Test
	public void validateOutPatientGetMedicationDetail() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getOutpatientMedication();
		String medicationDetail = medicationRepo.getMedicationDetail(medication.getPatientId(),
				medication.getPrescriptionId());
		System.out.println("Outpatient Medication Detail:\n" + medicationDetail);
		assertTrue(medicationDetail != null && medicationDetail.contains("Medication:"));
		assertTrue(medicationDetail.contains(IntegrationTestConfiguration
				.getMedicationDetailStringForOutpatientMedication()));
	}

	public void validateSig() throws OvidDomainException {
		Medication testMedication = IntegrationTestConfiguration.getMedicationForSig();
		String expectedSig = testMedication.getSig();
		OvidMedication retrievedMedication = loadMedicationByOrderId(testMedication);
		assertEquals(expectedSig, retrievedMedication.getSigDetail());

		testMedication = IntegrationTestConfiguration.getMedicationForSig2();
		expectedSig = testMedication.getSig();
		retrievedMedication = loadMedicationByOrderId(testMedication);
		assertEquals(expectedSig, retrievedMedication.getSigDetail());

		testMedication = IntegrationTestConfiguration.getMedicationForSig3();
		expectedSig = testMedication.getSig();
		retrievedMedication = loadMedicationByOrderId(testMedication);
		assertEquals(expectedSig, retrievedMedication.getSigDetail());
	}

	private OvidMedication loadMedicationByOrderId(Medication medication) throws OvidDomainException {
		String patientId = medication.getPatientId();
		String orderId = medication.getOrderNumber();

		Collection<OvidMedication> retrievedMedications = medicationRepo.getAllMedications(patientId);
		OvidMedication retrievedMedication = null;
		for (OvidMedication possibleMatch : retrievedMedications) {
			if (possibleMatch.getOrderNumber().equals(orderId))
				retrievedMedication = possibleMatch;
		}
		return retrievedMedication;
	}

	@Test
	public void validateGetAllMedicationsWithBadPatientId() throws OvidDomainException {
		Collection<OvidMedication> ovidMedications = null;
		ovidMedications = medicationRepo.getAllMedications(BAD_PATIENT_ID);
		assertTrue(ovidMedications.isEmpty());
	}

	@Test
	public void validateGetAllMedications() throws OvidDomainException {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy.MM.dd");
		System.out.println("Starting validateGetAllMedications");
		Collection<OvidMedication> ovidMedications = null;

		ovidMedications = medicationRepo.getAllMedications(IntegrationTestConfiguration
				.getTestPatientIdWithMultipleMedications().getId());
		for (OvidMedication med : ovidMedications) {
			System.out.println("medication name=[" + med.getMedication() + "] type =[" + med.getMedType()
					+ "] order id=[" + med.getOrderNumber() + "] refills=[" + med.getRefills() + "] status=["
					+ med.getStatus() + "] supply=[" + med.getSupply() + "] quantity=[" + med.getQuantity()
					+ "] prescription id=[" + med.getPrescriptionId() + "]");
			if (med.getStopDate() != null)
				System.out.println(" expires=[" + formatter.format(med.getStopDate()) + "]");
			if (med.getLastFilled() != null) {
				System.out.println(" last filled=[" + formatter.format(med.getLastFilled()) + "]");
			}
			if (med.getSigDetail() != null) {
				System.out.println(" sig=[" + med.getSigDetail() + "]");
			}
		}
		assertTrue(!ovidMedications.isEmpty());
		System.out.println("Ending validateGetAllMedications");
	}

	@Test
	public void validateInpatientNonVaMedicationsAreReturnedForPharmacyPatient() throws Exception {
		Medication medication = IntegrationTestConfiguration.getTestNonVAMedication();
		OvidMedication ovidMedication = findMedication(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull("Did not find medication " + medication.getPatientId(), ovidMedication);
	}

	@Test
	public void validateInpatientNonVaMedicationsStartDate() throws Exception {
		Medication medication = IntegrationTestConfiguration.getTestNonVAMedication();
		OvidMedication ovidMedication = findMedication(medication.getPatientId(), medication.getOrderNumber());
		assertEquals(null, ovidMedication.getStartDate());
	}

	@Test
	public void validateInpatientNonVaMedicationsStopDate() throws Exception {
		Medication medication = IntegrationTestConfiguration.getTestNonVAMedication();
		OvidMedication ovidMedication = findMedication(medication.getPatientId(), medication.getOrderNumber());
		assertEquals(null, ovidMedication.getStopDate());
	}

	@Test
	public void validateInpatientNonVaMedicationsActive() throws Exception {
		Medication medication = IntegrationTestConfiguration.getTestNonVAMedication();
		OvidMedication ovidMedication = findMedication(medication.getPatientId(), medication.getOrderNumber());
		assertEquals("ACTIVE", ovidMedication.getStatus());
	}
}
