package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.patient;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.PatientDataLayerOvid;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchCriteria;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;

public class PatientSearchTest {

	private Patient testPatient = null;

	@Before
	public void setup() {
		PatientTestHelper.initializeOvidConnectionManager();
		testPatient = IntegrationTestConfiguration.getTestPatient();
	}

	@Test
	public void searchPatientByLastName() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setPrototype(new Patient());
		criteria.getPrototype().setLastName(testPatient.getLastName());

		PatientDataLayer datalayer = new PatientDataLayerOvid();
		Patients patients = datalayer.patientSearch(criteria);
		assertNotNull(patients);
		assertEquals(1, patients.size());
	}

	@Test(expected = IllegalArgumentException.class)
	public void searchPatientInsufficientCriteria() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		PatientDataLayerOvid datalayer = new PatientDataLayerOvid();
		datalayer.patientSearch(criteria);
	}

	@Test
	public void searchWithNoResults() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setPrototype(new Patient());
		criteria.getPrototype().setLastName("XXX");

		PatientDataLayerOvid datalayer = new PatientDataLayerOvid();
		Patients patients = datalayer.patientSearch(criteria);
		assertNotNull(patients);
		assertEquals(0, patients.size());
	}

	@Test
	public void searchPatientBySsn() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		Patient prototype = new Patient();
		criteria.setPrototype(prototype);
		prototype.setSsn(testPatient.getSsn());
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

    @Test
    public void searchPatientByDashedSsn() {
        PatientSearchCriteria criteria = new PatientSearchCriteria();
        Patient prototype = new Patient();
        criteria.setPrototype(prototype);

        String testPatientSsn = testPatient.getSsn();
        String ssn1 = testPatientSsn.substring(0,3);
        String ssn2 = testPatientSsn.substring(3,5);
        String ssn3 = testPatientSsn.substring(5,9);
        testPatientSsn = String.format("%1s-%2s-%3s", ssn1, ssn2, ssn3);

        prototype.setSsn(testPatientSsn);
        PatientDataLayer dao = new PatientDataLayerOvid();
        Patients patients = dao.patientSearch(criteria);
        Patient patient = patients.get(0);
        assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
    }

    @Test
	public void searchPatientBySsnWithSearchString() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setSearchString(testPatient.getSsn());
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

	@Test
	public void searchPatientByInitialPlusLastFourSsn() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setLastInitialPlusLastFourOfSsn(extractInitialSsn(testPatient));
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

	private String extractInitialSsn(Patient patient) {
		return patient.getLastName().substring(0, 1) + patient.getSsn().substring(5);
	}

	@Test
	public void searchPatientByInitialPlusLastFourSsnWithSearchString() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setSearchString(extractInitialSsn(testPatient));
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

	@Test
	public void searchPatientByLastFourSsnWithSearchString() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setSearchString(extractInitialSsn(testPatient));
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

	@Test
	public void searchPatientByLastFourSsn() {
		PatientSearchCriteria criteria = new PatientSearchCriteria();
		criteria.setLastFourOfSsn(extractInitialSsn(testPatient));
		PatientDataLayer dao = new PatientDataLayerOvid();
		Patients patients = dao.patientSearch(criteria);
		Patient patient = patients.get(0);
		assertEquals(testPatient.getLastName().toLowerCase(), patient.getLastName().toLowerCase());
	}

}
