package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.medications;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.MedicationDataLayerOvid;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;
import com.medsphere.ovid.domain.ov.OvidDomainException;

public class MedicationDataLayerTest {
	private static final String BAD_PATIENT_ID = "BAD_PATIENT_ID";
	private static final String BAD_ORDER_ID = "BAD_PATIENT_ID";

	@Before
	public void setup() {
		PatientTestHelper.initializeOvidConnectionManager();
	}

	@Test
	public void validateGetMedicationLiteWithBadPatientId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationLite(BAD_PATIENT_ID, medication.getOrderNumber());
		//will get the order, but the patient id in the medication object will be wrong
		assertNotNull(retrievedMedication);
	}

	@Test
	public void validateGetMedicationLiteWithBadOrderId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationLite(medication.getPatientId(), BAD_ORDER_ID);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateGetMedicationDetailWithBadPatientId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationDetail(BAD_PATIENT_ID, medication.getOrderNumber());
		assertNull(retrievedMedication);
	}

	@Test
	public void validateGetMedicationDetailWithBadOrderId() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), BAD_ORDER_ID);
		assertNull(retrievedMedication);
	}

	@Test
	public void validateInpatientGetMedicationLite() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationLite(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals(retrievedMedication.getDrugName().substring(0,10), medication.getDrugName().substring(0,10));
		assertNotSame(retrievedMedication.getDrugName(), medication.getDrugName());
		assertNull(retrievedMedication.getMedicationDetail());
	}

//	@Test
	public void validateInpatientGetMedicationDetail() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getInpatientMedication2();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals(retrievedMedication.getDrugName(), medication.getDrugName());
		assertTrue(retrievedMedication.getMedicationDetail().contains("MEDICATION HISTORY for"));
		assertTrue(retrievedMedication.getMedicationDetail().contains(IntegrationTestConfiguration.getMedicationHistoryStringForInpatientMedication()));
		assertTrue(retrievedMedication.getMedicationDetail().contains(IntegrationTestConfiguration.getMedicationDetailStringForInpatientMedication()));
	}

	@Test
	public void validateOutpatientGetMedicationLite() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getOutpatientMedication();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationLite(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals(retrievedMedication.getDrugName().substring(0,10), medication.getDrugName().substring(0,10));
		assertNotSame(retrievedMedication.getDrugName(), medication.getDrugName());
		assertNull(retrievedMedication.getMedicationDetail());
	}

//	@Test
	public void validateOutpatientGetMedicationDetail() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getOutpatientMedication();
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals(retrievedMedication.getDrugName(), medication.getDrugName());
		assertTrue(retrievedMedication.getMedicationDetail().contains("MEDICATION HISTORY for"));
		assertTrue(retrievedMedication.getMedicationDetail().contains(IntegrationTestConfiguration.getMedicationHistoryStringForOutpatientMedication()));
		System.err.println(retrievedMedication.getMedicationDetail());
		System.out.println(IntegrationTestConfiguration.getMedicationDetailStringForOutpatientMedication());
		assertTrue(retrievedMedication.getMedicationDetail().contains(IntegrationTestConfiguration.getMedicationDetailStringForOutpatientMedication()));
	}

	@Ignore
	public void validateMedicationsAreReturnedForTestPatient() throws Exception {
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Patient patient = IntegrationTestConfiguration.getTestPatient();
		Medications medications = dao.getMedications(patient.getId(), true);
		assertTrue(medications.size() > 0);
	}

	@Test
	public void validateEmptyMedicationsAreReturnedForPatientWithNoData() throws Exception {
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Patient patient = IntegrationTestConfiguration.getTestPatientWithoutData();
		Medications medications = dao.getMedications(patient.getId(), true);
		assertTrue(medications.size() == 0);
	}

	@Ignore
	public void validateDiscreteMedationDatailsReturned() throws Exception {
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Patient patient = IntegrationTestConfiguration.getTestPatient();
		Medications medications = dao.getMedications(patient.getId(), true);
		assertTrue(medications.size() > 0);

		for (Medication medication : medications) {
			assertNotNull(medication.getDrugName());
			assertNotNull(medication.getStartDate());
		}

	}

	@Ignore
	public void validateActiveMedationDatailsReturned() throws Exception {
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Patient patient = IntegrationTestConfiguration.getTestPatient();
		Medications medications = dao.getMedications(patient.getId(), true);
		assertTrue(medications.size() > 0);

		for (Medication medication : medications) {
			assertNotNull(medication.getDrugName());
			assertNotNull(medication.getStartDate());
			assertTrue(medication.isActive());
		}

	}

	@Ignore
	public void validateMedicationDoesNotContainEllipsis() throws Exception {
		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Patient patient = IntegrationTestConfiguration.getTestPatient();
		Medications medications = dao.getMedications(patient.getId(), true);
		assertTrue(medications.size() > 0);

		for (Medication medication : medications) {
			assertTrue("'" + medication.getDrugName() + "' contains '..'", !medication.getDrugName().contains(".."));
		}
	}

	@Ignore
	public void activeMedicationsIncludeActiveSuspended() throws OvidDomainException {
		// Medication medication =
		// IntegrationTestConfiguration.getInpatientMedication2();
		Medication medication = new Medication();
		medication.setPatientId("205");
		medication.setOrderNumber("33985");

		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals("ACTIVE/SUSP", retrievedMedication.getStatus());
		assertEquals(true, retrievedMedication.isActive());
	}

	@Ignore
	public void medicationExpired1MonthIsActive() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getTestExpiredMedication();

		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		DateTime dt = new DateTime(medication.getEndDate());
		dt = dt.plusDays(30);
		dao.setCurrentDate(dt.toDate());
		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), medication.getOrderNumber());

		assertNotNull(retrievedMedication);
		assertEquals("EXPIRED", retrievedMedication.getStatus());
		assertEquals(true, retrievedMedication.isActive());
	}
	
	@Ignore
	public void medicationExpired6MonthsIsNotActive() throws OvidDomainException {
		Medication medication = IntegrationTestConfiguration.getTestExpiredMedication();

		MedicationDataLayerOvid dao = new MedicationDataLayerOvid();
		DateTime dt = new DateTime(medication.getEndDate());
		dt = dt.plusDays(180);
		dao.setCurrentDate(dt.toDate());

		Medication retrievedMedication = dao.getMedicationDetail(medication.getPatientId(), medication.getOrderNumber());
		assertNotNull(retrievedMedication);
		assertEquals("EXPIRED", retrievedMedication.getStatus());
		assertEquals(false, retrievedMedication.isActive());
	}
}
