package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.labs;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.lab.MicrobiologyLabDataLayerOvid;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterImpl;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.MutableDateFilter;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;
import com.agilex.healthcare.utility.DateHelper;

public class MicroLabDataLayerTest {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MicroLabDataLayerTest.class);

	private DateOnlyComparator dateOnlyComparator = new DateOnlyComparator();
	private LabResults retrievedResults;

	@Before
	public void setup() {
		PatientTestHelper.initializeOvidConnectionManager();
		Patient testPatient = IntegrationTestConfiguration.getLabPatient();
		retrievedResults = retrieveLabResultsForPatient(testPatient.getPatientIdentifier());

		logger.debug(String.format("retrieved %s lab(s) for patient %s", retrievedResults.size(), testPatient.getId()));
		for (LabResult result : retrievedResults) {
			logger.debug(String.format("result: [test.name=%s][test.id=%s][specimen=%s][dt=%s]", result.getTestname(), result.getTestId(), result.getSpecimenName(), result.getResultedDate()));
		}
	}

	private LabResults retrieveLabResultsForPatient(PatientIdentifier patientIdentifier) {
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/1990"));
		dateFilter.setEndDate(new Date());

		MicrobiologyLabDataLayerOvid datalayer = new MicrobiologyLabDataLayerOvid();
		LabResults results = datalayer.getLabResults(patientIdentifier, dateFilter);
		return results;
	}

	private LabResult filterLabFromList(LabResults results, LabResult resultToMatchOn) {
		LabResult matchedResult = null;
		for (LabResult potentialMatch : results) {
			logger.debug(String.format("compare [%s][%s] == [%s][%s]", resultToMatchOn.getResultedDate(), resultToMatchOn.getTestId(), potentialMatch.getResultedDate(), potentialMatch.getTestId()));
			if (dateOnlyComparator.compare(resultToMatchOn.getResultedDate(), potentialMatch.getResultedDate()) == 0 && potentialMatch.getTestId().contentEquals(resultToMatchOn.getTestId())) {
				logger.debug("found match");
				matchedResult = potentialMatch;
				break;
			}
		}
		return matchedResult;
	}

	@Test
	public void dateLayerReturnsLabs() {
		assertTrue(retrievedResults.size() >= 1);
	}

	@Test
	public void verifyTextValue() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getMicroLab();
		LabResult retrievedResult = filterLabFromList(retrievedResults, expectedLabResult);

		assertEquals(normalizeTextValue(expectedLabResult.getValue()), normalizeTextValue(retrievedResult.getValue()));
	}

	private String normalizeTextValue(String s) {
		s = s.replace("\n", "");
		s = s.replace(" ", "");
		return s;
	}

	class DateOnlyComparator implements Comparator<java.util.Date> {
		public int compare(java.util.Date d1, java.util.Date d2) {
			Calendar c1 = Calendar.getInstance();
			c1.setTime(d1);
			Calendar c2 = Calendar.getInstance();
			c2.setTime(d2);
			if (c1.get(Calendar.YEAR) != c2.get(Calendar.YEAR))
				return c1.get(Calendar.YEAR) - c2.get(Calendar.YEAR);
			return c1.get(Calendar.DAY_OF_YEAR) - c2.get(Calendar.DAY_OF_YEAR);
		}
	}

}
