package com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.labs;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.lab.LabDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.ovid.lab.ChemistryLabDataLayerOvid;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterImpl;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.MutableDateFilter;
import com.agilex.healthcare.testutility.IntegrationTestConfiguration;
import com.agilex.healthcare.testutility.PatientTestHelper;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class LabDataLayerTest {
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(LabDataLayerTest.class);

	private Patient testPatient;
	private DateOnlyComparator dateOnlyComparator = new DateOnlyComparator();
	private LabResults testPatientRetrievedResults;

	@Before
	public void setup() {
		PatientTestHelper.initializeOvidConnectionManager();
		testPatient = IntegrationTestConfiguration.getLabPatient();
		testPatientRetrievedResults = retrieveLabResultsForPatient(testPatient.getPatientIdentifier());

		logger.debug(String.format("retrieved %s lab(s) for patient %s", testPatientRetrievedResults.size(), testPatient.getId()));
		for (LabResult result : testPatientRetrievedResults) {
			logger.debug(String.format("result: [test.name=%s][test.id=%s][specimen=%s][order=%s][dt=%s][value=%s][refLow=%s][refHigh=%s]", result.getTestname(), result.getTestId(),
					result.getSpecimenName(), result.getOrderId(), result.getResultedDate(), result.getValue(), result.getReferenceLow(), result.getReferenceHigh()));
		}
	}

	@Test
	public void dateLayerReturnsLabs() {
		assertTrue(testPatientRetrievedResults.size() >= 1);
	}

	@Test
	public void ensureEachLabResultsContainTestId() {
		assertTrue(testPatientRetrievedResults.size() >= 1);
		for (LabResult result : testPatientRetrievedResults) {
			logger.debug(String.format("result: [test.name=%s][test.id=%s][specimen=%s][order=%s][dt=%s][value=%s][refLow=%s][refHigh=%s]", result.getTestname(), result.getTestId(),
					result.getSpecimenName(), result.getOrderId(), result.getResultedDate(), result.getValue(), result.getReferenceLow(), result.getReferenceHigh()));
			assertTrue(NullChecker.isNotNullish(result.getTestId()));
		}
	}

	@Test
	public void ensureCorrectReferenceRange() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLab();
		LabResult retrievedResult = filterLabFromList(testPatientRetrievedResults, expectedLabResult);
		assertNotNull(retrievedResult);
		assertEquals(expectedLabResult.getReferenceLow(), retrievedResult.getReferenceLow());
		assertEquals(expectedLabResult.getReferenceHigh(), retrievedResult.getReferenceHigh());
	}

	@Test
	public void validateGetLabResultsByTestId() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLab();
		PatientIdentifier patientId = testPatient.getPatientIdentifier();
		String labTestId = expectedLabResult.getTestId();
		ChemistryLabDataLayerOvid dataLayer = new ChemistryLabDataLayerOvid();
		LabResults results = dataLayer.getLabResultsByTestId(patientId, labTestId);
		assertTrue(results.size() > 0);
		boolean foundExpectedResult = false;
		for (LabResult result : results) {
			assertEquals(expectedLabResult.getTestId(), result.getTestId());
			if (dateOnlyComparator.compare(expectedLabResult.getResultedDate(), result.getResultedDate()) == 0 && result.getValue().equals(expectedLabResult.getValue())) {
				foundExpectedResult = true;
				break;
			}
		}
		assertTrue(foundExpectedResult);
	}

	@Test
	public void validateGetLabResultsByTestIdAndSpecimenId() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLab();
		PatientIdentifier patientId = testPatient.getPatientIdentifier();
		String labTestId = expectedLabResult.getTestId();
		String specimenId = expectedLabResult.getSpecimenId();
		LabDataLayer dataLayer = new ChemistryLabDataLayerOvid();
		LabResults results = dataLayer.getLabResultsByTestIdAndSpecimenId(patientId, labTestId, specimenId);
		assertTrue(results.size() > 0);
		boolean foundExpectedResult = false;
		for (LabResult result : results) {
			assertEquals(expectedLabResult.getTestId(), result.getTestId());
			assertEquals(expectedLabResult.getSpecimenId(), result.getSpecimenId());
			if (dateOnlyComparator.compare(expectedLabResult.getResultedDate(), result.getResultedDate()) == 0 && result.getValue().equals(expectedLabResult.getValue())) {
				foundExpectedResult = true;
				break;
			}
		}
		assertTrue(foundExpectedResult);
	}

	@Test
	public void validateLabResultByTestIdMaxReferenceRange() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLabForReferenceRangeCheck();
		PatientIdentifier patientId = testPatient.getPatientIdentifier();
		String labTestId = expectedLabResult.getTestId();
		LabDataLayer dataLayer = new ChemistryLabDataLayerOvid();
		LabResults results = dataLayer.getLabResultsByTestId(patientId, labTestId);
		assertTrue(results.size() > 0);
		LabResult matchedResult = null;
		for (LabResult result : results) {
			System.out.println("high=" + result.getReferenceHigh());
			System.out.println(result.getResultedDate());
			System.out.println(result.getValue());
			if (dateOnlyComparator.compare(expectedLabResult.getResultedDate(), result.getResultedDate()) == 0 && result.getValue().equals(expectedLabResult.getValue())) {
				matchedResult = result;
				break;
			}
		}
		assertNotNull(matchedResult);

		assertEquals(expectedLabResult.getReferenceLow(), matchedResult.getReferenceLow());
	}

	@Test
	public void validateLabResultByTestIdMinReferenceRange() {
		LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLabForReferenceRangeCheck();
		PatientIdentifier patientId = testPatient.getPatientIdentifier();
		String labTestId = expectedLabResult.getTestId();
		LabDataLayer dataLayer = new ChemistryLabDataLayerOvid();
		LabResults results = dataLayer.getLabResultsByTestId(patientId, labTestId);
		assertTrue(results.size() > 0);
		LabResult matchedResult = null;
		for (LabResult result : results) {
			System.out.println("low=" + result.getReferenceLow());
			System.out.println(result.getResultedDate());
			System.out.println(result.getValue());
			if (dateOnlyComparator.compare(expectedLabResult.getResultedDate(), result.getResultedDate()) == 0 && result.getValue().equals(expectedLabResult.getValue())) {
				matchedResult = result;
				break;
			}
		}
		assertNotNull(matchedResult);
		assertEquals(expectedLabResult.getReferenceLow(), matchedResult.getReferenceLow());
	}

	@Test
	public void validateLabResultByTestIdMinReferenceRangeFails() {
		if (IntegrationTestConfiguration.getTestSystem().equals("ov")) {
			LabResult expectedLabResult = IntegrationTestConfiguration.getNormalLab();
			PatientIdentifier patientId = testPatient.getPatientIdentifier();
			String labTestId = expectedLabResult.getTestId();
			LabDataLayer dataLayer = new ChemistryLabDataLayerOvid();
			LabResults results = dataLayer.getLabResultsByTestId(patientId, labTestId);
			assertTrue(results.size() > 0);
			LabResult matchedResult = null;
			for (LabResult result : results) {
				System.out.println("low=" + result.getReferenceLow());
				System.out.println(result.getResultedDate());
				System.out.println(result.getValue());
				if (dateOnlyComparator.compare(expectedLabResult.getResultedDate(), result.getResultedDate()) == 0 && result.getValue().equals(expectedLabResult.getValue())) {
					matchedResult = result;
					break;
				}
			}
			assertNotNull(matchedResult);
			assertNotSame(expectedLabResult.getReferenceLow(), matchedResult.getReferenceLow());
		}
	}

	private LabResults retrieveLabResultsForPatient(PatientIdentifier patientIdentifier) {
		MutableDateFilter dateFilter = new DateFilterImpl();
		dateFilter.setStartDate(DateHelper.parseDate("1/1/2001"));
		dateFilter.setEndDate(new Date());

		LabDataLayer datalayer = new ChemistryLabDataLayerOvid();
		LabResults results = datalayer.getLabResults(patientIdentifier, dateFilter);
		return results;
	}

	private LabResult filterLabFromList(LabResults results, LabResult resultToMatchOn) {
		LabResult matchedResult = null;
		for (LabResult potentialMatch : results) {
			logger.debug(String.format("compare [%s][%s] == [%s][%s]", resultToMatchOn.getResultedDate(), resultToMatchOn.getTestId(), potentialMatch.getResultedDate(), potentialMatch.getTestId()));
			if (dateOnlyComparator.compare(resultToMatchOn.getResultedDate(), potentialMatch.getResultedDate()) == 0 && potentialMatch.getTestId().contentEquals(resultToMatchOn.getTestId())) {
				logger.debug("found match");
				matchedResult = potentialMatch;
				break;
			}
		}
		return matchedResult;
	}

	class DateOnlyComparator implements Comparator<java.util.Date> {
		public int compare(java.util.Date d1, java.util.Date d2) {
			Calendar c1 = Calendar.getInstance();
			c1.setTime(d1);
			Calendar c2 = Calendar.getInstance();
			c2.setTime(d2);
			if (c1.get(Calendar.YEAR) != c2.get(Calendar.YEAR))
				return c1.get(Calendar.YEAR) - c2.get(Calendar.YEAR);
			return c1.get(Calendar.DAY_OF_YEAR) - c2.get(Calendar.DAY_OF_YEAR);
		}
	}

}
