package com.agilex.healthcare.mobilehealthplatform.datalayer.kmr;

import gov.hhs.fha.nhinc.adapter.fact.FactQueryRequestType;
import gov.hhs.fha.nhinc.adapter.fact.FactQueryResponseType;
import gov.hhs.fha.nhinc.adapter.fact.FactType;
import gov.hhs.fha.nhinc.adapter.fact.NameFactType;
import gov.hhs.fha.nhinc.adapter.fact.PatientSearchPayloadType;
import gov.hhs.fha.nhinc.adapter.fact.PersonFactType;
import gov.hhs.fha.nhinc.adapter.fact.ValueType;
import gov.hhs.fha.nhinc.adapter.factservice.CommonDataLayerFactServiceImpl;

import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.patient.PatientSearchCriteria;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.utility.DOBDate;
import com.agilex.healthcare.utility.NullChecker;

public class PatientDataLayerOpenVistaFactService implements PatientDataLayer {

	private static final String SSN_CODE = "2.16.840.1.113883.4.1";
	private static final String PRPA_TE201305UV02 = "PRPA_TE201305UV02";
	private static final String PRPA_IN201305UV02 = "PRPA_IN201305UV02";
	private static final String MOBILE_HEALTH_PLATFORM = "Mobile Health Platform";

	@Override
	public Patients patientSearch(PatientSearchCriteria criteria) {

		FactQueryRequestType factQueryRequest = createFactQueryRequest(criteria);
		
		FactQueryResponseType factQeuryResponse = invokeFactService(factQueryRequest);
		
		Patients patients = extractPatients(factQeuryResponse);

		return patients;
	}

	private Patients extractPatients(FactQueryResponseType factQeuryResponse) {
		Patients patients = new Patients();
		
		if (factQeuryResponse != null && factQeuryResponse.getFacts() != null && factQeuryResponse.getFacts().size() > 0){
			
			for ( FactType fact : factQeuryResponse.getFacts()) {
				PersonFactType personFact = (PersonFactType) fact;
				Patient patient = new Patient();
				patient.setLastName(personFact.getLegalName().getFamilyName());
				patient.setFirstName(personFact.getLegalName().getFirstName());
				patient.setMiddleInitial(personFact.getLegalName().getMiddleName());
				patient.setDateOfBirth(new DOBDate(personFact.getDateOfBirth()));
				patient.setGender(personFact.getGender().getCode());
				
				for (ValueType id : personFact.getId()) {
					if (SSN_CODE.equals(id.getCodeSystem())){
						patient.setId(id.getValue());
						patient.setSsn(id.getValue());
						break;
					}
				}	
				patients.add(patient);
			}
		} 
		
		return patients;
	}

	protected FactQueryResponseType invokeFactService(FactQueryRequestType factQueryRequest) {
		FactQueryResponseType factQueryResponse;
		
		try {
			factQueryResponse = CommonDataLayerFactServiceImpl.getInstance().getPatientFacts(factQueryRequest);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		return factQueryResponse;
	}

	@Override
	public PatientDemographics getDemographics(String id) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Patient getPatient(String id) {
		// TODO Auto-generated method stub
		return null;
	}
	
	protected FactQueryRequestType createFactQueryRequest(PatientSearchCriteria criteria){
		FactQueryRequestType factQueryRequest = new FactQueryRequestType();
		String queryId = UUID.randomUUID().toString();
		factQueryRequest.setQueryId(queryId );
		String senderId = MOBILE_HEALTH_PLATFORM;
		factQueryRequest.setSenderId(senderId);
		String interationId = PRPA_IN201305UV02;
		factQueryRequest.setInteractionId(interationId );
		String triggerEventCode = PRPA_TE201305UV02;
		factQueryRequest.setTriggerEventCode(triggerEventCode );
		PatientSearchPayloadType payload = new PatientSearchPayloadType();
		
		// Check name
		if (criteria.getPrototype() != null){
			Patient patientCriteria = criteria.getPrototype();
			NameFactType name = new NameFactType();
			
			if (NullChecker.isNotNullish(patientCriteria.getLastName()) || NullChecker.isNotNullish(patientCriteria.getFirstName())){
				
				if (NullChecker.isNotNullish(patientCriteria.getLastName())){
					name.setFamilyName(patientCriteria.getLastName());
				}
				
				if (NullChecker.isNotNullish(patientCriteria.getFirstName())){
					name.setFirstName(patientCriteria.getFirstName());
				}
				payload.setName(name);
			}
			
			
			if (NullChecker.isNotNullish(patientCriteria.getSsn())){
			    ValueType ssn = new ValueType();
			    ssn.setValue(patientCriteria.getSsn());
			    String ssnCode = SSN_CODE;
				ssn.setCodeSystem(ssnCode);
				payload.setId(ssn);
			}

		}
		
		factQueryRequest.setPatientSearchPayload(payload );
		return factQueryRequest;
	}

}
