package com.agilex.healthcare.mobilehealthplatform.security;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.net.MalformedURLException;

import javax.ws.rs.core.MediaType;

import junit.framework.Assert;

import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.security.oauth2.common.DefaultOAuth2AccessToken;
import org.springframework.security.oauth2.common.OAuth2AccessToken;

import com.agilex.healthcare.mobilehealthplatform.client.AuthenticationInfo;
import com.agilex.healthcare.mobilehealthplatform.dto.HAOauth2Authentication;
import com.agilex.healthcare.mobilehealthplatform.dto.TokenValidationRequest;
import com.agilex.healthcare.utility.ModeHelper;
import com.gargoylesoftware.htmlunit.BrowserVersion;
import com.gargoylesoftware.htmlunit.FailingHttpStatusCodeException;
import com.gargoylesoftware.htmlunit.WebClient;
import com.sun.jersey.api.client.Client;

public class VeteranOauthSecurityTest {

	private OauthClient oauthClient = new OauthClient();
	private String username = "zztest.patient01";
	private String password = "pass";
	private String MHP_BASE_URL = "http://localhost:8080/MobileHealthPlatformWeb";
	private String authorizeUrl = "http://localhost:8080/ssoeproxy/veteran/authorize";
	private String tokenUrl = "http://localhost:8080/AuthorizationServices/oauth/token";
	private String tokenValidationUri = "http://localhost:8080/AuthorizationServices/rest/validateToken";
	private String deleteTokenUrl = "http://localhost:8080/AuthorizationServices/rest/token";
	private String logoutUrl = "http://localhost:8080/ssoeproxy/logout";
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInCIHMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
	}
	
	@Test
	public void authenticateWithAuthorizationCode() throws Exception {
		validateAuthenticationWithOauth("intTest", "pass");
	}

	@Test
	public void testValidateAccessToken() throws Exception {
		OAuth2AccessToken accessToken = retrieveAccessToken("intTest", "pass", null);
		TokenValidationRequest tokenValidationRequest = new TokenValidationRequest();
		tokenValidationRequest.setToken(accessToken.getValue());
		HAOauth2Authentication haOauth2Authentication = validateToken(tokenValidationRequest);
		assertEquals("D123401", haOauth2Authentication.getUserName());
		assertEquals("intTest", haOauth2Authentication.getAuthorizatioRequest().getClientId());
		assertEquals("D123401", haOauth2Authentication.getMhpUser().getId());
		assertEquals("zztest", haOauth2Authentication.getMhpUser().getPatient().getFirstName());
		assertEquals("patient01", haOauth2Authentication.getMhpUser().getPatient().getLastName());
		assertTrue(haOauth2Authentication.getAuthorities().toString().contains("ROLE_CONSUMER"));
		assertNotNull(haOauth2Authentication.getLastLoginDate());
	}
	
	private HAOauth2Authentication validateToken(TokenValidationRequest tokenValidationRequest) {
		Client restClient = new com.sun.jersey.api.client.Client();
		HAOauth2Authentication haOauth2Authentication = restClient.resource(tokenValidationUri).accept(MediaType.APPLICATION_XML).post(HAOauth2Authentication.class,tokenValidationRequest);
		return haOauth2Authentication;
	}
	
	
	@Test
	public void validateLaunchPadWorksWithOauth() throws Exception {
		validateAuthenticationWithOauth("launchPad", "LAUNCHPAD");
	}

	@Test
	public void invalidAccessTokeDoesNotAllowAccessToData() throws Exception {
		String clientId = "intTest";
		String clientPassword = "pass";

		DefaultOAuth2AccessToken accessToken = (DefaultOAuth2AccessToken) retrieveAccessToken(clientId, clientPassword, null);

		accessToken = accessToken.setValue(accessToken.getValue() + "X");
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.UNAUTHORIZED);
	}

	@Test
	public void verifyDeleteTokenDeniesAccessToResources() throws Exception {
		String clientId = "intTest";
		String clientPassword = "pass";
		WebClient userAgent = new WebClient(BrowserVersion.CHROME);

		OAuth2AccessToken accessToken = retrieveAccessToken(clientId, clientPassword, userAgent);
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.OK);
		oauthClient.deleteToken(deleteTokenUrl, accessToken);
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.UNAUTHORIZED);
	}
	

	@Test
	public void verifyLogoffInvalidatesOauthTokensAndLaunchesRedirectUri() throws Exception {
		String clientId = "intTest";
		String clientPassword = "pass";
		WebClient userAgent = new WebClient(BrowserVersion.CHROME);

		OAuth2AccessToken accessToken = retrieveAccessToken(clientId, clientPassword, userAgent);
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.OK);
		oauthClient.deleteToken(deleteTokenUrl, accessToken);
		logoff(userAgent, "http://someananoymousUri");
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.UNAUTHORIZED);
	}


	private void validateAuthenticationWithOauth(String clientId, String clientPassword) throws Exception {
		OAuth2AccessToken accessToken = retrieveAccessToken(clientId, clientPassword, null);
		assertDataRetrieve(oauthClient, accessToken, HttpStatus.OK);
	}

	private OAuth2AccessToken retrieveAccessToken(String clientId, String clientPassword, WebClient userAgent) throws Exception {
		AuthenticationInfo authInfo = new AuthenticationInfo();
		authInfo.setUsername(username);
		authInfo.setPassword(password);
		authInfo.setClientId(clientId);
		authInfo.setClientSecret(clientPassword);
		authInfo.setExpectAuthenticationRequired(true);
		authInfo.setAuthorizeUrl(authorizeUrl);
		authInfo.setTokenUrl(tokenUrl);

		OAuth2AccessToken accessToken = retrieveAccessToken(authInfo, userAgent);

		return accessToken;
	}

	private OAuth2AccessToken retrieveAccessToken(AuthenticationInfo authInfo, WebClient userAgent) throws Exception {

		AuthorizationResult authorizationResult = oauthClient.authorize(authInfo, userAgent);
		OAuth2AccessToken accessToken = oauthClient.getAccessToken(authInfo, authorizationResult);

		return accessToken;
	}

	private void assertDataRetrieve(OauthClient client, OAuth2AccessToken accessToken, HttpStatus expectedHttpStatus) {
		HttpHeaders headers = new HttpHeaders();
		headers.set("Authorization", String.format("%s %s", OAuth2AccessToken.BEARER_TYPE, accessToken.getValue()));
		assertEquals(expectedHttpStatus, client.getStatusCode(MHP_BASE_URL + "/rest/patient/EDIPI/D123401", headers));
	}

	private void logoff(WebClient userAgent, String redirectUri) throws IOException, MalformedURLException {

		try {
				userAgent.getPage(logoutUrl + "?redirect_uri=" + redirectUri);
				Assert.fail("You should not get here, as it should be redirected");
		} catch (FailingHttpStatusCodeException e) {
			String actualRedirectUrl = e.getResponse().getResponseHeaderValue("Location");
			assertEquals(redirectUri, actualRedirectUrl);
		}
	}
}
