package com.agilex.healthcare.mobilehealthplatform.mdws.translator;

import java.util.List;
import java.util.UUID;

import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsDateTranslationUtil;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfTaggedVitalSignArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfTaggedVitalSignSetArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfVitalSignSetTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfVitalSignTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedVitalSignArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedVitalSignArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedVitalSignSetArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedVitalSignSetArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.VitalSignSetTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.VitalSignTO;
import com.agilex.healthcare.utility.NullChecker;

public class VitalsTranslator {

	public VitalEntries translateArrays(TaggedVitalSignArrays taggedVitalSignArrays) {
		if (taggedVitalSignArrays == null)
			return new VitalEntries();

		return unwrapArrayOfTaggedVitalSignArray(taggedVitalSignArrays.getArrays());
	}

	private VitalEntries unwrapArrayOfTaggedVitalSignArray(ArrayOfTaggedVitalSignArray arrays) {
		if (arrays == null)
			return new VitalEntries();

		return unwrapTaggedVitalSignArrays(arrays.getTaggedVitalSignArray());
	}

	private VitalEntries unwrapTaggedVitalSignArrays(List<TaggedVitalSignArray> taggedVitalSignArray) {
		VitalEntries vitalEntries = new VitalEntries();

		if (taggedVitalSignArray == null || taggedVitalSignArray.size() == 0)
			return vitalEntries;

		for (TaggedVitalSignArray vitalSignArray : taggedVitalSignArray) {
			vitalEntries.addAll(unwrapArrayOfVitalSigns(vitalSignArray.getVitals()));
		}

		return vitalEntries;
	}

	private VitalEntries unwrapArrayOfVitalSigns(ArrayOfVitalSignTO arrayOfVitalSign) {
		if (arrayOfVitalSign == null)
			return new VitalEntries();

		return unwrapListOfVitalSigns(arrayOfVitalSign.getVitalSignTO());
	}

	private VitalEntries unwrapListOfVitalSigns(List<VitalSignTO> vitalSignTos) {
		VitalEntries vitalEntries = new VitalEntries();

		if (vitalSignTos == null || vitalSignTos.size() == 0)
			return vitalEntries;

		for (VitalSignTO vitalSignTo : vitalSignTos) {
			vitalEntries.add(constructVitalEntry(vitalSignTo));
		}

		return vitalEntries;
	}

	private VitalEntry constructVitalEntry(VitalSignTO vitalSignTo) {
		VitalEntry vitalEntry = new VitalEntry();

		setDateValues(vitalEntry, vitalSignTo.getTimestamp());
		setVitalObservationValues(vitalEntry, vitalSignTo);
		vitalEntry.setUniqueId(vitalSignTo.getId());
		vitalEntry.setSourceSystem(vitalSignTo.getFacility().getText());

		return vitalEntry;
	}

	// /

	public VitalEntries translateSets(TaggedVitalSignSetArrays vitalSigns) {
		if (vitalSigns == null)
			return new VitalEntries();

		return unwrapArrayOfTaggedVitalSignArray(vitalSigns.getArrays());
	}

	private VitalEntries unwrapArrayOfTaggedVitalSignArray(ArrayOfTaggedVitalSignSetArray arrayOfTaggedVitalSignSetArray) {
		if (arrayOfTaggedVitalSignSetArray == null)
			return new VitalEntries();

		return unwrapListOfTaggedVitalSignArray(arrayOfTaggedVitalSignSetArray.getTaggedVitalSignSetArray());
	}

	private VitalEntries unwrapListOfTaggedVitalSignArray(List<TaggedVitalSignSetArray> list) {
		VitalEntries vitalEntries = new VitalEntries();

		if (list == null || list.size() == 0)
			return vitalEntries;

		for (TaggedVitalSignSetArray taggedVitalSignSetArray : list) {
			vitalEntries.addAll(unwrapTaggedVitalSignArray(taggedVitalSignSetArray));
		}

		return vitalEntries;
	}

	private VitalEntries unwrapTaggedVitalSignArray(TaggedVitalSignSetArray taggedVitalSignArray) {
		if (taggedVitalSignArray == null)
			return new VitalEntries();

		return unwrapArrayOfVitalSignSetTO(taggedVitalSignArray.getSets());
	}

	private VitalEntries unwrapArrayOfVitalSignSetTO(ArrayOfVitalSignSetTO arrayOfVItalSignSetTO) {
		if (arrayOfVItalSignSetTO == null)
			return new VitalEntries();

		return unwrapListOfVitalSignSets(arrayOfVItalSignSetTO.getVitalSignSetTO());
	}

	private VitalEntries unwrapListOfVitalSignSets(List<VitalSignSetTO> listOfVitalSignSet) {
		VitalEntries vitalEntries = new VitalEntries();

		if (listOfVitalSignSet == null || listOfVitalSignSet.size() == 0)
			return vitalEntries;

		for (VitalSignSetTO vitalSignSetTO : listOfVitalSignSet) {
			vitalEntries.addAll(unwrapListOfVitalSigns(vitalSignSetTO));
		}

		return vitalEntries;
	}

	private VitalEntries unwrapListOfVitalSigns(VitalSignSetTO vitalSignSetTO) {
		VitalEntries vitalEntries = new VitalEntries();
		ArrayOfVitalSignTO arrayOfVitalSignTo = vitalSignSetTO.getVitalSigns();

		List<VitalSignTO> listOfVitalSigns = arrayOfVitalSignTo.getVitalSignTO();
		if (listOfVitalSigns == null || listOfVitalSigns.size() == 0)
			return vitalEntries;

		for (VitalSignTO vitalSignTO : listOfVitalSigns) {
			// setting the qualifier from VitalSignSetTo to VitalEtnry
			vitalSignTO.setQualifiers(vitalSignSetTO.getQualifiers());

			vitalEntries.add(constructVitalEntry(vitalSignTO, vitalSignSetTO));
		}

		return vitalEntries;
	}

	private VitalEntry constructVitalEntry(VitalSignTO vitalSign, VitalSignSetTO vitalSignSetTO) {
		VitalEntry vitalEntry = new VitalEntry();

		setIdentifier(vitalSign, vitalEntry);

		// setting any entries from VitalSignSetTO to vitalEntry
		setDateValues(vitalEntry, vitalSignSetTO.getTimestamp());
		vitalEntry.setSourceSystem(vitalSignSetTO.getFacility().getTag());

		setVitalObservationValues(vitalEntry, vitalSign);

		return vitalEntry;
	}

	private void setIdentifier(VitalSignTO vitalSign, VitalEntry vitalEntry) {
		// Handle scenarios where vitalEntry doesn't have a data identifier.
		// This is a temporary fix for pilot by Ryan for domains that don't use
		// the self URI
		if (NullChecker.isNullish(vitalSign.getType().getId())) {
			vitalEntry.setUniqueId(UUID.randomUUID().toString());
		} else {
			vitalEntry.setUniqueId(vitalSign.getType().getId());
		}
	}

	private void setVitalObservationValues(VitalEntry vitalEntry, VitalSignTO vitalSign) {

		if (VitalsCode.BLOOD_PRESSURE.equalsIgnoreCase(vitalSign.getType().getName())) {
			vitalEntry.setSection(VitalSectionCode.BloodPressure);
			setBloodPressureObservationValues(vitalEntry, vitalSign);
			return;
		}

		String type = vitalSign.getType().getName();
		VitalObservation vitalObservation = new VitalObservation();

		if (VitalsCode.PULSE.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.HEART_BEAT);
			vitalEntry.setSection(VitalSectionCode.Pulse);
		} else if (VitalsCode.TEMPERATURE.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.BODY_TEMPERATURE);
			vitalEntry.setSection(VitalSectionCode.Temperature);
		} else if (VitalsCode.HEIGHT.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.HEIGHT);
			vitalEntry.setSection(VitalSectionCode.Height);
		} else if (VitalsCode.WEIGHT.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.WEIGHT);
			vitalEntry.setSection(VitalSectionCode.Weight);
		} else if (VitalsCode.PAIN.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.PAIN);
			vitalEntry.setSection(VitalSectionCode.Pain);
		} else if (VitalsCode.RESPIRATION.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.RESPIRATION);
			vitalEntry.setSection(VitalSectionCode.Respiration);
		} else if (VitalsCode.PULSE_OXIMETRY.equalsIgnoreCase(type) || VitalsCode.PULSE_OXIMETRY_MISSPELLED.equalsIgnoreCase(type)) {
			vitalObservation.setType(ObservationTypeCode.PULSE_OXIMETRY);
			vitalEntry.setSection(VitalSectionCode.PulseOximetry);
		} else {
			// logger.debug("Vital type from CDW was not translated properly");
		}

		vitalObservation.setVitalEntry(vitalEntry);
		vitalObservation.setValue(vitalSign.getValue1());
		vitalEntry.addObservation(vitalObservation);
	}

	void setBloodPressureObservationValues(VitalEntry vitalEntry, VitalSignTO vitalSign) {
		String[] systolicDiastolic = vitalSign.getValue1().split("/");

		if (systolicDiastolic.length == 2) {
			VitalObservation observation = new VitalObservation();
			observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC);
			observation.setValue(systolicDiastolic[0]);
			observation.setValueUnits("mmHg");
			vitalEntry.addObservation(observation);

			observation = new VitalObservation();
			observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC);
			observation.setValue(systolicDiastolic[1]);
			observation.setValueUnits("mmHg");
			vitalEntry.addObservation(observation);
		}
	}

	void setDateValues(VitalEntry vitalEntry, String timestamp) {
		if (timestamp == null)
			return;

		MdwsDateTranslationUtil dateTranslator = new MdwsDateTranslationUtil();
		vitalEntry.setEntryDate(dateTranslator.extractDateOrNull(timestamp));
	}

}
