package com.agilex.healthcare.mobilehealthplatform.mdws.translator;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsDateTranslationUtil;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfPatientTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfTaggedPatientArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.PatientTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedPatientArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedPatientArrays;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class PatientTranslator {

	public Patients translateFromTaggedPatientArrays(TaggedPatientArrays mdwsTaggedPatientArrays) {
		if (mdwsTaggedPatientArrays != null) {
			ArrayOfTaggedPatientArray mdwsArrayOfTaggedPatientArray = mdwsTaggedPatientArrays.getArrays();
			PatientTranslator translator = new PatientTranslator();
			return translator.unwrapArrayOfTaggedPatientArray(mdwsArrayOfTaggedPatientArray, translator);
		}
		return new Patients();
	}

	Patients unwrapArrayOfTaggedPatientArray(ArrayOfTaggedPatientArray mdwsArrayOfTaggedPatientArray, PatientTranslator translator) {
		if (mdwsArrayOfTaggedPatientArray != null) {
			List<TaggedPatientArray> mdwsTaggedPatientArrays = mdwsArrayOfTaggedPatientArray.getTaggedPatientArray();
			Patients patients = new Patients();
			for (TaggedPatientArray taggedPatientArray : mdwsTaggedPatientArrays) {
				ArrayOfPatientTO mdwsArrayOfPatients = taggedPatientArray.getPatients();
				patients.addAll(translator.unwrapArrayOfPatientTO(mdwsArrayOfPatients, translator));
			}
			return patients;
		}
		return new Patients();
	}

	Patients unwrapArrayOfPatientTO(ArrayOfPatientTO mdwsArrayOfPatients, PatientTranslator translator) {
		Patients patients = new Patients();
		if (mdwsArrayOfPatients != null) {
			List<PatientTO> mdwsPatients = mdwsArrayOfPatients.getPatientTO();
			for (PatientTO mdwsPatient : mdwsPatients) {
				Patient patient = constructPatient(mdwsPatient);
				patients.add(patient);
			}
			return patients;
		}

		return new Patients();
	}

	public Patient constructPatient(PatientTO mdwsPatient) {
		Patient patient = new Patient();
		patient.setDisplayName(mdwsPatient.getPatientName());
		patient.setGender(mdwsPatient.getGender());
		patient.setSsn(mdwsPatient.getSsn());

		PatientIdentifier patientIdentifier = new PatientIdentifier("default", mdwsPatient.getLocalPid());
		patient.setPatientIdentifier(patientIdentifier);

		PatientTranslator translator = new PatientTranslator();
		translator.tryToSetDateOfBirth(patient, mdwsPatient);
		translator.tryToSetPatientAge(patient, mdwsPatient);
		translator.tryToSetPatientNames(patient, mdwsPatient);
		translator.tryToSetPatientMiddleInitial(patient);

		return patient;
	}

	void tryToSetPatientAge(Patient patient, PatientTO mdwsPatient) {
		try {
			int age = mdwsPatient.getAge();
			if (age == 0) {
				long calculatedAge = calculateAge(patient.getDateOfBirth().getDate());
				patient.setAge(String.valueOf(calculatedAge));
			}
		} catch (Exception e) {}

	}

	public static long calculateAge(Date dob) {
		return DateHelper.calculateDeltaInYears(dob, new Date());
	}

	public void tryToSetPatientNames(Patient patient, PatientTO mdwsPatient) {
		// Assuming "LAST,FIRST MIDDLE" Syntax
		String patientName = mdwsPatient.getPatientName();
		if (NullChecker.isNotNullish(patientName)) {
			String[] splitNames = patientName.trim().split(",");
			patient.setLastName(splitNames[0]);
			String[] firstAndMiddle = splitNames[1].toString().trim().split(" ");
			switch (firstAndMiddle.length) {
			case 1:
				patient.setFirstName(firstAndMiddle[0]);
				break;
			case 2:
				patient.setFirstName(firstAndMiddle[0]);
				patient.setMiddleName(firstAndMiddle[1]);
				break;
            case 3: // rare, but possible
                patient.setFirstName(firstAndMiddle[0]);
                patient.setMiddleName(firstAndMiddle[1] + " " + firstAndMiddle[2]);
                break;
            case 4: // even more rare, but possible
                patient.setFirstName(firstAndMiddle[0]);
                patient.setMiddleName(firstAndMiddle[1] + " " + firstAndMiddle[2] + " " + firstAndMiddle[3]);
                break;
            default:
                patient.setFirstName(firstAndMiddle[0]); // first name is required by vista;
			}
		}
	}

	void tryToSetPatientMiddleInitial(Patient patient) {
		String middleName = patient.getMiddleName();
		if (NullChecker.isNotNullish(middleName)) {
			char initial = middleName.charAt(0);
			patient.setMiddleInitial(Character.toString(initial).toUpperCase());
		}
	}

	void tryToSetDateOfBirth(Patient patient, PatientTO mdwsPatient) {
		MdwsDateTranslationUtil util = new MdwsDateTranslationUtil();
		Date dobDate = util.extractDateOrNull(mdwsPatient.getDob());
		if (dobDate == null)
			patient.setDateOfBirth(null);
		else
			patient.setDateOfBirth(new DOBDate(dobDate));
	}

}
