package com.agilex.healthcare.mobilehealthplatform.mdws.translator;

import java.util.*;

import org.apache.log4j.Logger;

import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsDateTranslationUtil;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfMedicationTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.ArrayOfTaggedMedicationArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.MedicationTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedMedicationArray;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedMedicationArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedText;
import com.agilex.healthcare.utility.NullChecker;

public class MedicationTranslator {
	private static final Logger LOGGER = Logger.getLogger(MedicationTranslator.class);
	
	public Medications translateFromTaggedMedicationArrays(TaggedMedicationArrays mdwsTaggedMedicationArrays) {
		if(mdwsTaggedMedicationArrays == null)
			return new Medications();
		
		return unwrapArrayOfTaggedMedicationArray(mdwsTaggedMedicationArrays.getArrays());
	}
	
	private Medications unwrapArrayOfTaggedMedicationArray(ArrayOfTaggedMedicationArray taggedMedicationArray) {
		Medications medications = new Medications();
		
		if(taggedMedicationArray == null)
			return medications;
		
		return unwrapTaggedMedicationArrayList(taggedMedicationArray.getTaggedMedicationArray());
	}
	
	private Medications unwrapTaggedMedicationArrayList(List<TaggedMedicationArray> taggedMedicationArrayList) {
		Medications medications = new Medications();
		
		if(taggedMedicationArrayList == null || taggedMedicationArrayList.size() == 0)
			return medications;
		
		for(TaggedMedicationArray mdwsMedicationArray : taggedMedicationArrayList) {
			medications.addAll(unwrapArrayOfMedication(mdwsMedicationArray.getMeds()));
		}
		 
		return medications;
	}
	
	private Medications unwrapArrayOfMedication(ArrayOfMedicationTO mdwsArrayOfMedication) {
		Medications medications = new Medications();
		
		if(mdwsArrayOfMedication == null)
			return medications;
		
		return unwrapMedicationTOList(mdwsArrayOfMedication.getMedicationTO());
	}
	

	private Medications unwrapMedicationTOList(List<MedicationTO> mdwsMedications) {
		Medications medications = new Medications();
		
		if(mdwsMedications == null || mdwsMedications.size() == 0)
			return medications;
		
		for(MedicationTO mdwsMedication : mdwsMedications) {
			medications.add(constructMedication(mdwsMedication));
		}
		
		return medications;
	}
	
	public Medication constructMedication(MedicationTO mdwsMedication) {
		assert mdwsMedication != null;
		
		Medication medication = new Medication();

		medication.setId(getId(mdwsMedication));
		medication.setUniqueId(getId(mdwsMedication));
		medication.setOrderNumber(mdwsMedication.getOrderId());
		medication.setDrugName(mdwsMedication.getName());
		medication.setPrescriptionId(mdwsMedication.getRxNum());
		medication.setMedicationDetail(mdwsMedication.getDetail());
		medication.setStatus(mdwsMedication.getStatus());
		medication.setQuantity(mdwsMedication.getQuantity());
		medication.setSig(mdwsMedication.getSig());
		medication.setRefills(mdwsMedication.getRefills());
		medication.setDaysSupply(mdwsMedication.getDaysSupply());
		medication.setCategory(mdwsMedication.getType());
		
		medication.setTotalRefills(parseStringAsInteger(mdwsMedication.getRefills()));
		medication.setRefillsRemaining(parseStringAsInteger((mdwsMedication.getRemaining())));
		
		medication.setLastFilledDate(parseDate(mdwsMedication.getLastFillDate()));
		medication.setStartDate(parseDate(mdwsMedication.getIssueDate()));
		medication.setEndDate(parseDate(mdwsMedication.getStopDate()));
		medication.setExpiredDate(parseDate(mdwsMedication.getExpirationDate()));
		
		tryToSetMedicationSource(medication, mdwsMedication);
		setSourceSystem(medication, mdwsMedication);
		
		return medication;
	}
	
	private void setSourceSystem(Medication medication, MedicationTO mdwsMedication) {
		TaggedText facility = mdwsMedication.getFacility();
		if(facility == null) return;

        String source = TranslationHelper.alterFacilityName(facility.getText());
		
		medication.setSourceSystem(source);
	}
	
	private String getId(MedicationTO medicationTo) {
		String medicationId = medicationTo.getId();
		if(NullChecker.isNullish(medicationId))
			medicationId = medicationTo.getOrderId();
		if(NullChecker.isNullish(medicationId))
			medicationId = UUID.randomUUID().toString();
		
		return medicationId;
		
	}

	private static void tryToSetMedicationSource(Medication medication, MedicationTO mdwsMedication) {
		if (mdwsMedication.isIsInpatient()) {
			medication.setMedicationSource(MedicationFilter.INPATIENT);
		} else {
			medication.setMedicationSource(MedicationFilter.OUTPATIENT);
		}
	}
	
	private static Integer parseStringAsInteger(String value) {
		Integer refills = null;

		if (NullChecker.isNotNullish(value)) {
			try {
				refills = Integer.parseInt(value);
			} catch (NumberFormatException e) {
				LOGGER.debug("Error parsing remaining refills, setting to null", e);
			}
		}
		
		return refills;
	}
	
	private static Date parseDate(String value) {
		MdwsDateTranslationUtil util = new MdwsDateTranslationUtil();
		return util.extractDateOrNull(value);
	}
}
