package com.agilex.healthcare.mobilehealthplatform.mdws.translator;

import java.util.*;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsDateTranslationUtil;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.*;
import com.agilex.healthcare.utility.*;

public class ChemistryLabTranslator {

	public LabResults translate(TaggedChemHemRptArrays taggedChemHemReportArrays) {
		if (taggedChemHemReportArrays == null)
			return new LabResults();

		return unwrap(taggedChemHemReportArrays.getArrays());
	}

	private LabResults unwrap(ArrayOfTaggedChemHemRptArray arrayOfTaggedChemHemReportArray) {
		if (arrayOfTaggedChemHemReportArray == null)
			return new LabResults();

		return unwrapReportArrays(arrayOfTaggedChemHemReportArray.getTaggedChemHemRptArray());
	}

	private LabResults unwrapReportArrays(List<TaggedChemHemRptArray> taggedChemHemReportArrays) {
		LabResults results = new LabResults();
		if (taggedChemHemReportArrays == null || taggedChemHemReportArrays.size() == 0)
			return new LabResults();

		for (TaggedChemHemRptArray taggedChemHemReporttArray : taggedChemHemReportArrays) {
			results.addAll(unwrap(taggedChemHemReporttArray));
		}

		return results;
	}

	private LabResults unwrap(TaggedChemHemRptArray taggedChemHemReportArray) {
		if (taggedChemHemReportArray == null)
			return new LabResults();

		return unwrapArrayOfChemReport(taggedChemHemReportArray.getRpts());
	}

	private LabResults unwrapArrayOfChemReport(ArrayOfChemHemRpt arrayOfChemReport) {
		if (arrayOfChemReport == null)
			return new LabResults();

		return unwrapReports(arrayOfChemReport.getChemHemRpt());
	}

	private LabResults unwrapReports(List<ChemHemRpt> chemHemReports) {
		LabResults results = new LabResults();

		if (chemHemReports == null || chemHemReports.size() == 0)
			return results;

		for (ChemHemRpt chemHemReport : chemHemReports) {
			results.addAll(unwrap(chemHemReport));
		}

		return results;
	}
	
	private LabResults unwrap(ChemHemRpt chemHemReport) {
		if(chemHemReport == null)
			return new LabResults();
		
		return unwrapArray(chemHemReport.getResults(), chemHemReport);
	}
	
	private LabResults unwrapArray(ArrayOfLabResultTO results, ChemHemRpt chemHemReport) {
		if(results == null)
			return new LabResults();
		
		return unwrapResults(results.getLabResultTO(), chemHemReport);
	}
	
	private LabResults unwrapResults(List<LabResultTO> listOfLabResults, ChemHemRpt chemHemReport) {
		LabResults labResults = new LabResults();
		
		if(listOfLabResults == null || listOfLabResults.size() == 0)
			return labResults;
		
		for(LabResultTO labResult : listOfLabResults) {
			labResults.add(constructLabResult(labResult, chemHemReport));
		}
		
		return labResults;
	}

	private LabResult constructLabResult(LabResultTO labResultTo, ChemHemRpt chemHemReport) {
		LabResult labResult = new LabResult();
		
		setTestData(labResult, labResultTo);
		setSpeciminData(labResult, chemHemReport.getSpecimen());
		labResult.setValue(labResultTo.getValue());
		labResult.setIndicator(labResultTo.getBoundaryStatus());
		labResult.setAbnormalIndicatorTriFlag(determineTriStateValue(labResultTo.getBoundaryStatus()));
		labResult.setResultedDate(extractDate(chemHemReport.getTimestamp()));
		
		return labResult;
	}
	
	void setTestData(LabResult labResult, LabResultTO labResultTo) {
		LabTestTO test = labResultTo.getTest();
		labResult.setUniqueId(test.getId());
		labResult.setTestId(test.getId());
		labResult.setTestname(test.getName());
		labResult.setDisplayName(test.getName());
		labResult.setDisplayDescription(test.getName());
		labResult.setValueUnits(test.getUnits());
		labResult.setLoinc(test.getLoinc());
		
		parseAndSetReferenceRange(labResult, test);
	}

	void setSpeciminData(LabResult labResult, LabSpecimenTO specimen) {
		if (specimen != null) {
			labResult.setSpecimenId(specimen.getId());
			labResult.setSpecimenName(specimen.getName());
			labResult.setAccessionNumber(specimen.getAccessionNum());

            TaggedText facility = specimen.getFacility();
            if(facility != null) {
                labResult.setSourceSystem(facility.getTag()); // Tag will be the Site id
            }
        }
	}
	
	private void parseAndSetReferenceRange(LabResult labResult, LabTestTO mdwsLabTest) {
		String referenceRange = mdwsLabTest.getRefRange();
		if (NullChecker.isNotNullish(referenceRange)) {
			String[] lowAndHigh = referenceRange.split("-");
			if (lowAndHigh != null && lowAndHigh.length == 2) {
				labResult.setReferenceLow(lowAndHigh[0].trim());
				labResult.setReferenceHigh(lowAndHigh[1].trim());
			}
		} else {
			labResult.setReferenceLow(mdwsLabTest.getLowRef());
			labResult.setReferenceHigh(mdwsLabTest.getHiRef());
		}
	}

	private Date extractDate(String date) {
		MdwsDateTranslationUtil dateTranslator = new MdwsDateTranslationUtil();
		return dateTranslator.extractDateOrNull(date);
	}

	private TriState determineTriStateValue(String boundaryStatus) {
		if (NullChecker.isNullish(boundaryStatus))
			return TriState.FALSE;
		else if (boundaryStatus.contains("H") || boundaryStatus.contains("L"))
			return TriState.TRUE;
		else
			return TriState.UNKNOWN;
	}
}