package com.agilex.healthcare.mobilehealthplatform.mdws.connection;

import java.io.Serializable;
import java.net.URL;
import java.util.Date;
import java.util.UUID;

import javax.xml.ws.BindingProvider;

import org.apache.cxf.endpoint.Client;
import org.apache.cxf.frontend.ClientProxy;
import org.apache.cxf.transport.http.HTTPConduit;
import org.apache.cxf.transports.http.configuration.HTTPClientPolicy;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;

import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.EmrSvc;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.EmrSvcSoap;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfigurationHelper;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.VistaSite;
import com.agilex.healthcare.mobilehealthplatform.security.AppUser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class EmrConnection implements Serializable {

	private static final long serialVersionUID = -7675397237993685048L;
	private static final Log logger = LogFactory.getLog(EmrConnection.class);
	private String id;
	private MdwsConfiguration configuration;
	private EmrSvcSoap port;
	private Date refreshedTimestamp;

	public EmrConnection() {
		this.id = UUID.randomUUID().toString();
	}
	
	public EmrConnection(EmrSvcSoap port) {
		this.id = UUID.randomUUID().toString();
		this.port = port;
	}
	
	// this constructor will be removed once connection pool is adjusted
	@Deprecated
	public EmrConnection(MdwsConfiguration configuration) {
		this.id = UUID.randomUUID().toString();
		this.configuration = configuration;
		this.refreshedTimestamp = new Date();
	}
	
	public EmrConnection(MdwsConfiguration configuration, String siteCode, boolean authenticateFlag) {
		this.id = UUID.randomUUID().toString();
		this.configuration = configuration;
		this.port = establishConnection(configuration, siteCode, authenticateFlag);
		this.refreshedTimestamp = new Date();
	}

	public Date getRefreshedTimestamp() {
		return refreshedTimestamp;
	}

	public void refreshTimestamp() {
		this.refreshedTimestamp = new Date();
	}

	public String getId() {
		return id;
	}

	public EmrSvcSoap getPort() {
		return port;
	}
	
	public void disconnect(){
		try {
			this.getPort().disconnect();
			logger.debug("Disconnected");
		} catch (Exception e) {
			logger.error(e);
		}
	}
	
	public MdwsConfiguration getConfiguration() {
		return this.configuration;
	}
	
	private EmrSvcSoap establishConnection(MdwsConfiguration configuration, String siteCode, boolean authenticated) {
		EmrSvcSoap port = null;
		
		// Check for CDW
		if (MdwsConfigurationHelper.CDW_SITECODE.equalsIgnoreCase(siteCode) || 
				MdwsConfigurationHelper.CDW_RAW_SITECODE.equalsIgnoreCase(siteCode)){
			port = getEmrSvcSoap(configuration.getEndPointUrl(), configuration.getWsConnectTimeout(), configuration.getWsRequestTimeout());
			VistaSite site = configuration.getVistaSite(siteCode);
			port.connect(siteCode);
			if (authenticated){
				port.login(site.getAccessCode(), site.getVerifyCode(), "");
			}
		}else{
			if(authenticated){
				MhpUser mhpUser = getMhpUser();
				String url = mhpUser.getMdwsInfo().getBaseUrl() + "/" + configuration.getEndPointUrl();
				port = getEmrSvcSoap(url, configuration.getWsConnectTimeout(), configuration.getWsRequestTimeout());
				setSessionOnPort(port, mhpUser.getMdwsInfo().getSessionId());
			}else{
				String url = configuration.getEndPointBaseUrl() + "/" + configuration.getEndPointUrl();
				port = getEmrSvcSoap(url, configuration.getWsConnectTimeout(), configuration.getWsRequestTimeout());
			}
		}
		
		return port;
	}
	
	private EmrSvcSoap getEmrSvcSoap(String endPointUrl, int connectionTimeOut, int requestTimeout){
		URL wsdlLocation = constructWsdlUrl(endPointUrl);
		EmrSvc emrService = new EmrSvc(wsdlLocation);
		EmrSvcSoap port = emrService.getEmrSvcSoap();
		
		((BindingProvider)port).getRequestContext().put(BindingProvider.SESSION_MAINTAIN_PROPERTY, true);
		((BindingProvider)port).getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, endPointUrl);	    
		((BindingProvider) port).getRequestContext().put("javax.xml.ws.client.connectionTimeout", connectionTimeOut);
		((BindingProvider) port).getRequestContext().put("javax.xml.ws.client.receiveTimeout", requestTimeout);
		
		return port;
	}
		
	private URL constructWsdlUrl(String endPointUrl){
		String wsdlLocation = endPointUrl + "?WSDL";
		URL wsdlLocationURL = null;
		try {
			wsdlLocationURL = new URL(wsdlLocation);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		
		return wsdlLocationURL;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		EmrConnection other = (EmrConnection) obj;
		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;
		return true;
	}
	
	private  MhpUser getMhpUser(){
        SecurityContext ctx = SecurityContextHolder.getContext();

        AppUser appUser = (AppUser) ctx.getAuthentication().getPrincipal();
        MhpUser mhpUser = appUser.getMhpUser();
		mhpUser.setUserName(appUser.getUsername());

        return mhpUser;
	}
	
	private void setSessionOnPort(EmrSvcSoap port, String sessionId) {
		Client client = ClientProxy.getClient(port);

		HTTPClientPolicy policy = new HTTPClientPolicy();
		HTTPConduit httpConduit = (HTTPConduit)client.getConduit();
		
		policy.setCookie("ASP.NET_SessionId=" + sessionId);
		
		httpConduit.setClient(policy);
	}
	
}
