package com.agilex.healthcare.mobilehealthplatform.datalayer.securemessage;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.mhvservice.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MhvMdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.*;
import com.agilex.healthcare.utility.NullChecker;

public class SecureMessageDataLayerMdws extends MdwsMhvDataLayer implements SecureMessageDataLayer {
	
	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(SecureMessageDataLayerMdws.class);

	private MhvMdwsConfiguration mhvMdwsConfiguration;

	public SecureMessageDataLayerMdws(MhvMdwsConfiguration mhvMdwsConfiguration) {
		this.mhvMdwsConfiguration = mhvMdwsConfiguration;
	}

    private MhvServiceSoap getPort() {
        return getConnection(mhvMdwsConfiguration);
    }

    @Override
	public SecureMessageUser fetchSecureMessageUser(String userId) {
		SmUserTO mhvUser = fetchSmUser(userId, SecureMessageCode.ID_TYPE_ICN);
		
		return SecureMessageUserTranslator.translateMhvSmUser(mhvUser);
	}
	
	@Override
	public SecureMessageFolders fetchFolders(String userId) {
		SmUserTO mhvUser = fetchSmUser(userId, SecureMessageCode.ID_TYPE_SMID);
		
		return SecureMessageFolderTranslator.translateMhvArrayOfFolderTo(mhvUser.getMailbox());
	}

	@Override
	public SecureMessageContacts fetchContacts(String userId) {
		SmUserTO mhvUser = fetchSmUser(userId, SecureMessageCode.ID_TYPE_SMID);
		
		return SecureMessageContactTranslator.translateMhvArrayOfTriageGroup(mhvUser.getGroups());
	}
	
	private SmUserTO fetchSmUser(String userId, String idType) {
        MhvServiceSoap port = getPort();
		SmUserTO mhvUser = port.getSmUser(getMdwsPasswordField(), userId, idType);
		MdwsResponseValidator.validateMdwsObject(mhvUser);
		
		return mhvUser;
	}

	@Override
	public SecureMessages fetchSecureMessages(String userId, SecureMessageFilter secureMessageFilter) {
		Integer folderId = Integer.parseInt(secureMessageFilter.getFolder());
		int pageStart = (Integer.parseInt(secureMessageFilter.getPageNumber()) - 1) * Integer.parseInt(secureMessageFilter.getPageSize());
		int pageEnd = (Integer.parseInt(secureMessageFilter.getPageNumber())) * Integer.parseInt(secureMessageFilter.getPageSize());

        MhvServiceSoap port = getPort();
		SecureMessageThreadsTO mhvSecureMessages = port.getSecureMessages(getMdwsPasswordField(), Integer.parseInt(userId), folderId, pageStart, pageEnd);
		MdwsResponseValidator.validateMdwsObject(mhvSecureMessages);

		return SecureMessageTranslator.translateMhvSecureMessageThreads(mhvSecureMessages);
	}

	@Override
	public SecureMessage readSecureMessage(String userId, SecureMessage secureMessage) {
		Integer addresseeId = determineSecureMessageFieldId(secureMessage.getAddresseeId());
		Integer addresseeOplock = determineSecureMessageFieldId(secureMessage.getAddresseeOplock());

        MhvServiceSoap port = getPort();
		MessageTO readMessage = port.readMessage(getMdwsPasswordField(), addresseeId, addresseeOplock);
		MdwsResponseValidator.validateMdwsObject(readMessage);

		return SecureMessageTranslator.toSecureMessage(readMessage);
	}

	@Override
	public SecureMessage sendSecureMessage(String userId, SecureMessage secureMessage) {
		Integer mailGroup = determineSecureMessageFieldId(secureMessage.getMailGroup());
		Integer subjectCategory = determineSecureMessageFieldId(secureMessage.getSubjectCategory());
		Integer fromId = determineSecureMessageFieldId(secureMessage.getFromId());
		Integer toId = determineSecureMessageFieldId(secureMessage.getToId());

        MhvServiceSoap port = getPort();
		ThreadTO newMessageThread = port.sendNewMessage(getMdwsPasswordField(), secureMessage.getSubject(), mailGroup, 
				subjectCategory, fromId, toId, secureMessage.getMessageBody());
		MdwsResponseValidator.validateMdwsObject(newMessageThread);

		SecureMessages messages = SecureMessageTranslator.translateThread(newMessageThread);

		return messages.get(0);
	}

	@Override
	public SecureMessage replySecureMessage(String userId, SecureMessage secureMessage) {
        MhvServiceSoap port = getPort();
		MessageTO messageReply = port.sendReplyMessage(getMdwsPasswordField(), Integer.parseInt(secureMessage.getReplyToMessageId()), 
				Integer.parseInt(secureMessage.getFromId()), Integer.parseInt(secureMessage.getToId()), secureMessage.getMessageBody());
		MdwsResponseValidator.validateMdwsObject(messageReply);

		return SecureMessageTranslator.toSecureMessage(messageReply);
	}

	@Override
	public SecureMessage moveSecureMessage(String userId, String messageId, String folderId) {
		Integer secureMessageUserId = this.determineSecureMessageFieldId(userId);

        MhvServiceSoap port = getPort();
		MessageTO movedMessage = port.moveMessage(getMdwsPasswordField(), secureMessageUserId, Integer.parseInt(messageId), 
				Integer.parseInt(folderId));
		MdwsResponseValidator.validateMdwsObject(movedMessage);

		return SecureMessageTranslator.toSecureMessage(movedMessage);
	}

	@Override
	public SecureMessage saveSecureMessageDraft(String patientId, SecureMessage secureMessage) {
		Integer replyToMessageId = determineSecureMessageFieldId(secureMessage.getReplyToMessageId());
		Integer subjectCategoryId = determineSecureMessageFieldId(secureMessage.getSubjectCategory());
		Integer messageId = determineSecureMessageFieldId(secureMessage.getUniqueId());
		Integer messageOplockId = determineSecureMessageFieldId(secureMessage.getMessageOplock());
		Integer threadOplockId = determineSecureMessageFieldId(secureMessage.getThreadOplock());
		Integer fromId = determineSecureMessageFieldId(secureMessage.getFromId());
		Integer toId = determineSecureMessageFieldId(secureMessage.getToId());

        MhvServiceSoap port = getPort();
		ThreadTO savedDraft = port.saveSecureMessageDraft(getMdwsPasswordField(), replyToMessageId, secureMessage.getSubject(), 
				subjectCategoryId, messageId, fromId, toId, secureMessage.getMessageBody(), messageOplockId, threadOplockId);
		MdwsResponseValidator.validateMdwsObject(savedDraft);

		SecureMessages messages = SecureMessageTranslator.translateThread(savedDraft);

		return messages.get(0);
	}

	@Override
	public SecureMessage sendSecureMessageDraft(String patientId, SecureMessage secureMessage) {
        MhvServiceSoap port = getPort();
        // MDWS does not provide the ability to update a draft before sending; this must be done manually,
        // regardless of whether it is needed or not.
        SecureMessage savedDraft = saveSecureMessageDraft(patientId, secureMessage);

        Integer messageOplock = determineSecureMessageFieldId(savedDraft.getMessageOplock());
        Integer messageId = determineSecureMessageFieldId(savedDraft.getUniqueId());

		MessageTO sentDraft = port.sendSecureMessageDraft(getMdwsPasswordField(), messageId, messageOplock);
		MdwsResponseValidator.validateMdwsObject(sentDraft);
		
		return SecureMessageTranslator.toSecureMessage(sentDraft);
	}
	
	@Override
	public void deleteSecureMessageDraft(String userId, String messageId) {
		Integer secureMessageId = determineSecureMessageFieldId(messageId);

        MhvServiceSoap port = getPort();
		MessageTO deletedMessage = port.deleteDraft(getMdwsPasswordField(), secureMessageId);
		MdwsResponseValidator.validateMdwsObject(deletedMessage);
	}
	
	private Integer determineSecureMessageFieldId(String property) {
		if(NullChecker.isNotNullish(property)) {
			return Integer.parseInt(property);
		}
		logger.error("Secure Message Property was not set");
		return 0;
	}
	
	private String getMdwsPasswordField() {
		if(mhvMdwsConfiguration.isServiceConnection())
			return mhvMdwsConfiguration.getMdwsPwdWithService();
		else
			return mhvMdwsConfiguration.getMdwsPwdWithSid();
	}
	
}
