package com.agilex.healthcare.mobilehealthplatform.datalayer.radiology;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.connection.EmrConnection;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedRadiologyReportArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfigurationHelper;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.RadiologyTranslator;
import com.agilex.healthcare.utility.*;

public class RadiologyDataLayerMdws extends MdwsEmrDataLayer implements RadiologyDataLayer {


	public RadiologyDataLayerMdws(MdwsConfiguration mdwsConfiguration) {
		super(mdwsConfiguration);
	}
	
	@Override
	public RadiologyResults getResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		
		String siteCode = super.getSiteCodeFromPatientIdentifier(patientIdentifier);
		String siteType  =  new MdwsConfigurationHelper().extractSiteTypeFromAssigningAuthority(patientIdentifier.getAssigningAuthority());
		
		EmrConnection authenticatedConnection = super.getAuthenticatedConnectionWithPatientSelect(patientIdentifier);		
		
		String startDate = getStartDateAsString(dateFilter.getStartDate(), siteType);
		String endDate = getEndDateAsString(dateFilter.getEndDate(), siteType);
		
		TaggedRadiologyReportArrays taggedRadiologyReports = authenticatedConnection.getPort().getRadiologyReportsBySite(startDate, endDate, siteCode);
		MdwsResponseValidator.validateMdwsObject(taggedRadiologyReports);
		
		RadiologyResults results = translate(taggedRadiologyReports);
		return filter(results, dateFilter);
	}
	
	@Override
	public RadiologyResult fetchResult(PatientIdentifier patientIdentifier, String resultId) {
		RadiologyResults results = getResults(patientIdentifier, DateFilterFactory.createEmptyFilter());
		
		if(results == null)
			return null;
		
		return extract(results, resultId);
	}
	
	private RadiologyResult extract(RadiologyResults results, String resultId) {
		
		RadiologyResult found = null;
		for(RadiologyResult result : results) {
			if(NullSafeStringComparer.areEqualIgnoreCase(result.getUniqueId(), resultId)) {
				found = result;
			}
		}
		
		return found;
	}

	private RadiologyResults translate(TaggedRadiologyReportArrays taggedRadiologyReportArrays) {
		RadiologyTranslator translator = new RadiologyTranslator();
		return translator.translate(taggedRadiologyReportArrays);
	}
	
	private RadiologyResults filter(RadiologyResults results, DateFilter filter) {
		return DateFilterer.filterByDate(results, filter);
	}
	
	private String getStartDateAsString(Date date, String system) {
		MdwsDateTranslationUtil translator = new MdwsDateTranslationUtil();
		return translator.startDateBySystem(date, system);
	}

	private String getEndDateAsString(Date date, String system) {
		MdwsDateTranslationUtil translator = new MdwsDateTranslationUtil();
		return translator.endDateBySystem(date, system);
	}

}
