package com.agilex.healthcare.mobilehealthplatform.datalayer.radiology;

import java.util.Date;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.SourceSystemCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.connection.EmrConnection;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedRadiologyReportArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.RadiologyTranslator;

public class RadiologyDataLayerCdw extends CdwRawPatientDataLayer<RadiologyResult, RadiologyResults> implements RadiologyDataLayer {

	public RadiologyDataLayerCdw(MdwsConfiguration mdwsConfiguration) {
		super(mdwsConfiguration);
	}

	@Override
	public RadiologyResults getResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		EmrConnection authenticatedConnection = super.getAuthenticatedConnectionWithPatientSelect(patientIdentifier.getUniqueId());

		String startDate = getStartDateAsString(dateFilter.getStartDate(), SourceSystemCode.CDW);
		String endDate = getEndDateAsString(dateFilter.getEndDate(), SourceSystemCode.CDW);
		String siteCode = super.getSiteCodeFromPatientIdentifier(patientIdentifier);

		TaggedRadiologyReportArrays radiologyReportArrays = authenticatedConnection.getPort().getRadiologyReportsBySite(startDate, endDate, siteCode);
		super.returnConnection(authenticatedConnection);

		MdwsResponseValidator.validateMdwsObject(radiologyReportArrays);
		
		RadiologyResults radiologyResults = translateAndFilter(radiologyReportArrays, dateFilter);
		return super.setIdentifiers(radiologyResults, patientIdentifier, SourceSystemCode.CDW_RAW);
	}

	@Override
	public RadiologyResult fetchResult(PatientIdentifier patientIdentifier, String resultId) {
		RadiologyResults results = getResults(patientIdentifier, DateFilterFactory.createEmptyFilter());
		
		RadiologyResult search = null;
		for(RadiologyResult current : results) {
			if(current.getUniqueId().equals(resultId)) {
				search = current;
			}
		}
		
		return search;
	}

	private RadiologyResults translateAndFilter(TaggedRadiologyReportArrays radiologyReportArrays, DateFilter dateFilter) {
		RadiologyTranslator translator = new RadiologyTranslator();
		RadiologyResults translated =  translator.translate(radiologyReportArrays);
		return DateFilterer.filterByDate(translated, dateFilter);
	}

	private String getStartDateAsString(Date date, String system) {
		MdwsDateTranslationUtil translator = new MdwsDateTranslationUtil();
		return translator.startDateBySystem(date, system);
	}

	private String getEndDateAsString(Date date, String system) {
		MdwsDateTranslationUtil translator = new MdwsDateTranslationUtil();
		return translator.endDateBySystem(date, system);
	}

}
