package com.agilex.healthcare.mobilehealthplatform.datalayer.patient;

import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientDemographics;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.Patients;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsEmrDataLayer;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsResponseException;
import com.agilex.healthcare.mobilehealthplatform.mdws.connection.EmrConnection;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.PatientTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedPatientArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.DemographicTranslator;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.PatientTranslator;
import com.agilex.healthcare.utility.NullChecker;

public class PatientDataLayerMdws extends MdwsEmrDataLayer implements PatientDataLayer{

	private final static org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientDataLayerMdws.class);

	public PatientDataLayerMdws(MdwsConfiguration configuration) {
        super(configuration);
	}



    @Override
	public Patients patientSearch(PatientSearchCriteria criteria) {
		String searchString = criteria.determineSearchString();
//		logger.debug("Searching patients with criteria: " + searchString);

		String vistaSiteCode = criteria.getVistaSiteCode();
//		logger.debug("Searching patients with Vista SiteCode: " + vistaSiteCode);


		if (NullChecker.isNullish(searchString))
			throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_INPUT, "Insufficient search criteria");
		if (NullChecker.isNullish(vistaSiteCode))
			throw new MdwsResponseException(MdwsResponseException.UserMessages.INVALID_INPUT, "Vista Site Code is unavailabe to perform search against.");

		PatientTranslator translator = new PatientTranslator();
		EmrConnection connection = super.getAuthenticatedConnection(vistaSiteCode);
		TaggedPatientArrays taggedPatientArrays = connection.getPort().match(searchString);

		if (taggedPatientArrays.getFault() != null) {
			String errorMsg = taggedPatientArrays.getFault().getMessage();
			logger.error(errorMsg);
			throw new MdwsResponseException(taggedPatientArrays.getFault());
		}
		Patients translatedPatients =  translator.translateFromTaggedPatientArrays(taggedPatientArrays);

        Patients filteredPatients = applyNamePostFiltering(criteria, translatedPatients);
        return filteredPatients;

	}

    //MDWS returns more patients than we expect due to it's loose search criteria
    //ensure first and last names contain the appropriate search criteria
    protected Patients applyNamePostFiltering(PatientSearchCriteria criteria, Patients translatedPatients) {
        boolean discardPatient;

        String lastName = "";
        String firstName = "";

        //determine first and last name search criteria
        if (criteria.getPrototype() != null) {
            Patient prototype = criteria.getPrototype();

            if (NullChecker.isNotNullish(prototype.getLastName())){
                lastName = criteria.getPrototype().getLastName().toLowerCase();
            }
            if (NullChecker.isNotNullish(prototype.getFirstName())){
                firstName = criteria.getPrototype().getFirstName().toLowerCase();
            }
        }

        //should we apply the filters?
        Boolean applyLastNameFilter = NullChecker.isNotNullish(lastName);
        Boolean applyFirstNameFilter = NullChecker.isNotNullish(firstName);


        //apply filters if necessary
        Patients filteredPatients = new Patients();
        for (Patient translatedPatient : translatedPatients) {
            discardPatient = false;  //assume patient is good

            if (applyLastNameFilter && !translatedPatient.getLastName().toLowerCase().contains(lastName)){
                discardPatient = true;
            }
            else if (applyFirstNameFilter && !translatedPatient.getFirstName().toLowerCase().contains(firstName)){
                discardPatient = true;
            }

            if (!discardPatient){
                filteredPatients.add(translatedPatient);
            }
        }
        return filteredPatients;
    }

    @Override
	public PatientDemographics getDemographics(PatientIdentifier patientIdentifier) {

		EmrConnection authenticatedSite = super.getAuthenticatedConnectionWithPatientSelect(patientIdentifier);

		PatientTO mdwsPatient = authenticatedSite.getPort().getDemographics();

		if (mdwsPatient.getFault() != null) {
			String errorMsg = mdwsPatient.getFault().getMessage();
			logger.error(errorMsg);
			throw new RuntimeException(errorMsg);
		}
		DemographicTranslator translator = new DemographicTranslator();
		PatientDemographics patient = translator.translate(mdwsPatient.getDemographics());
		return patient;
	}

	@Override
	public Patient getPatient(PatientIdentifier patientIdentifier) {

		PatientTranslator translator = new PatientTranslator();
		EmrConnection authenticatedSite = super.getAuthenticatedConnectionWithPatientSelect(patientIdentifier);

		PatientTO mdwsPatient = authenticatedSite.getPort().select(patientIdentifier.getUniqueId());

		if (mdwsPatient.getFault() != null) {
			String errorMsg = mdwsPatient.getFault().getMessage();
			logger.error(errorMsg);
			throw new RuntimeException(errorMsg);
		}

		return translator.constructPatient(mdwsPatient);
	}
}
