package com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill;

import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillException;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsResponseException;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsResponseValidator;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.mhvservice.MedicationTO;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.mhvservice.MhvServiceSoap;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.mhvservice.TaggedMedicationArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MhvMdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MhvSoapHelper;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.MedicationRefillTranslator;
import com.agilex.system.health.SmokeTest;
import com.agilex.system.health.SmokeTestStatus;

public class MedicationRefillDataLayerMdws implements MedicationRefillDataLayer, SmokeTest {

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MedicationRefillDataLayerMdws.class);

    private final MhvMdwsConfiguration configuration;

	private final String SUCCESS_CODE = "OK";
	public final String ERROR_MESSAGE = "The request was either made too early or the system is not taking requests at this time. Please try again later.";

	public MedicationRefillDataLayerMdws(MhvMdwsConfiguration mhvConfiguration) {
        configuration = mhvConfiguration;
	}

    //TODO: Need connection management/pooling similar to EmrConnection
    private MhvServiceSoap getPort() {
        return MhvSoapHelper.getPort(configuration);
    }

    @Override
	public Medication refillMedication(Medication medication) {
		String siteCode = extractSiteFromAssigningAuthority(medication.getAssigningAuthority());
        MhvServiceSoap port = getPort();
		MedicationTO mdwsMedication = port.refillPrescription("ignored", siteCode, medication.getPatientId(), medication.getPrescriptionId());

        boolean passValidation = true;
        try {
		    MdwsResponseValidator.validateMdwsObject(mdwsMedication);
        } catch (MdwsResponseException mRE) {
            logError(mRE);
            passValidation = false;
        }

		MedicationRefillTranslator translator = new MedicationRefillTranslator();
		Medication refilledMedication = translator.createMedication(mdwsMedication);
		copyInformationMissingFromRefillResponse(medication, refilledMedication);

		if (!SUCCESS_CODE.equals(refilledMedication.getStatus()) || !passValidation) {
			generateValidationException(medication);
		}
		return refilledMedication;
	}

	@Override
	public Medications fetchMedications(PatientIdentifier patientIdentifier) {
		String siteCode = extractSiteFromAssigningAuthority(patientIdentifier.getAssigningAuthority());
        Medications translatedMedications = new Medications();
        try {
            MhvServiceSoap port = getPort();
            TaggedMedicationArrays mdwsTaggedMedicationArrays = port.getPrescriptionsHL7("ignored", siteCode, patientIdentifier.getUniqueId());
            MdwsResponseValidator.validateMdwsObject(mdwsTaggedMedicationArrays);

            MedicationRefillTranslator translator = new MedicationRefillTranslator();
            translatedMedications = translator.translateTaggedMedications(mdwsTaggedMedicationArrays);

            // Data tweaks prior to return
            determineRefillActions(translatedMedications);
            setPatientIdentifiers(translatedMedications, patientIdentifier);
        } catch (MdwsResponseException mRE) {
            // No medications exist,
            logError(mRE);
        }
		return translatedMedications;
	}

	private ValidationErrors<Medication> generateValidationException(Medication medication) {
		ValidationError validationError = new ValidationError(ERROR_MESSAGE, "Status");
		ValidationErrors<Medication> validationErrors = new ValidationErrors<Medication>();
		validationErrors.add(validationError);

		throw new MedicationRefillException(validationErrors);
	}

	private void determineRefillActions(Medications medications) {
		MedicationRefillHelper helper = new MedicationRefillHelper();
		helper.determineRefillAction(medications);
	}

	private void copyInformationMissingFromRefillResponse(Medication original, Medication refillResponse) {
		refillResponse.setOrderNumber(original.getOrderNumber());
		refillResponse.setDrugName(original.getDrugName());
	}

	private void setPatientIdentifiers(Medications medications, PatientIdentifier patientIdentifier) {
		for (Medication medication : medications) {
			medication.setPatientIdentifier(patientIdentifier);
		}
	}

    private void logError(MdwsResponseException mRE) {
        logger.error("MED REFILL ERROR - Response from MDWS");
    }

	String extractSiteFromAssigningAuthority(String assigningAuthority) {
		int length = assigningAuthority.length();
		return assigningAuthority.substring(length - 3, length);
	}

    @Override
    public SmokeTestStatus run() {
        SmokeTestStatus smokeTestStatus =  new SmokeTestStatus();

        try {
            MhvServiceSoap port  = getPort();
//            port.getVersion();
            smokeTestStatus.setSuccess(true);
        } catch (Exception e) {
            smokeTestStatus.setException(e);
        }

        smokeTestStatus.setConfiguration(this.configuration.toString());

        return smokeTestStatus;

    }
}
