package com.agilex.healthcare.mobilehealthplatform.datalayer.lab;

import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.NotImplementedException;

import com.agilex.healthcare.mobilehealthplatform.domain.LabResult;
import com.agilex.healthcare.mobilehealthplatform.domain.LabResults;
import com.agilex.healthcare.mobilehealthplatform.domain.LabTestGroups;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsDateTranslationUtil;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsEmrDataLayer;
import com.agilex.healthcare.mobilehealthplatform.mdws.MdwsResponseValidator;
import com.agilex.healthcare.mobilehealthplatform.mdws.connection.EmrConnection;
import com.agilex.healthcare.mobilehealthplatform.mdws.generatedwsdl.emrservice.TaggedChemHemRptArrays;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfiguration;
import com.agilex.healthcare.mobilehealthplatform.mdws.soapconsumer.MdwsConfigurationHelper;
import com.agilex.healthcare.mobilehealthplatform.mdws.translator.ChemistryLabTranslator;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullSafeStringComparer;

public class ChemistryLabDataLayerMdws extends MdwsEmrDataLayer implements ChemistryLabDataLayer {
	
	private static final String VISTA_START_DATE = "01/01/1980";

	public ChemistryLabDataLayerMdws(MdwsConfiguration mdwsConfiguration) {
		super(mdwsConfiguration);
	}
	
	@Override
	public LabResults getLabResults(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		EmrConnection authenticatedConnection = super.getAuthenticatedConnectionWithPatientSelect(patientIdentifier);
		
		MdwsDateTranslationUtil translationUtil = new MdwsDateTranslationUtil();
		String siteType  =  new MdwsConfigurationHelper().extractSiteTypeFromAssigningAuthority(patientIdentifier.getAssigningAuthority());
		//String siteType = "VISTA";
		String fromDate = translationUtil.dateFormatBySystem(fromDate(dateFilter.getStartDate()), siteType);
		String toDate = translationUtil.dateFormatBySystem(toDate(dateFilter.getEndDate()), siteType);
		
		TaggedChemHemRptArrays taggedChemHemReportArrays = authenticatedConnection.getPort().getChemHemReports(fromDate, toDate, 0);
		super.disconnectFromCdwIfApplicable(authenticatedConnection);
		
		MdwsResponseValidator.validateMdwsObject(taggedChemHemReportArrays);
		
		LabResults labResults = translate(taggedChemHemReportArrays);
		fixIdentifiers(labResults, siteType);
		return DateFilterer.filterByDate(labResults, dateFilter);
	}
	
	void fixIdentifiers(LabResults labResults, String system) {
		for(LabResult labResult : labResults) {
			labResult.setSystemId(system);
		}
	}

	@Override
	public LabResults getLabResultsByTestId(PatientIdentifier patientIdentifier, String testId) {
		LabResults labResults = getLabResults(patientIdentifier, DateFilterFactory.createEmptyFilter());
		return filterByTestId(labResults, testId);
	}
	
	LabResults translate(TaggedChemHemRptArrays taggedChemistryReportArrays) {
		ChemistryLabTranslator translator = new ChemistryLabTranslator();
		return translator.translate(taggedChemistryReportArrays);
	}
	
	LabResults filterByTestId(LabResults labResults, String testId) {
		LabResults filteredLabs = new LabResults();
		
		for(LabResult labResult : labResults) {
			if(NullSafeStringComparer.areEqual(testId, labResult.getTestId())) {
				filteredLabs.add(labResult);
			}
		}
		
		return filteredLabs;
	}
	
	public LabResults filterBySpecimenId(LabResults labResults, String specimenId) {
		LabResults filteredLabs = new LabResults();
		
		for(LabResult labResult : labResults) {
			if(NullSafeStringComparer.areEqual(specimenId, labResult.getSpecimenId())) {
				filteredLabs.add(labResult);
			}
		}
		
		return filteredLabs;
	}
	
	private Date fromDate(Date filterStartDate) {
		if(filterStartDate == null)
			return DateHelper.parseDate(VISTA_START_DATE);
		
		return filterStartDate;
	}
	
	private Date toDate(Date filterEndDate) {
		if(filterEndDate == null)
			return DateHelper.getToday();
		
		return filterEndDate;
	}

	@Override
	public LabResults getLabResultsLite(String patientId, DateFilter dateFilter) {
		throw new NotImplementedException();
	}
	
	@Override
	public LabTestGroups lookupLabGroupsByTestId(String labTestId) {
		throw new NotImplementedException();
	}

	@Override
	public Map<String, LabTestGroups> lookupLabGroupsByTestIds(List<String> labTestIds) {
		throw new NotImplementedException();
	}

}
