package com.agilex.healthcare.mobilehealthplatform.datalayer.vitals;

import java.util.ArrayList;
import java.util.List;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Mapper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.SimpleEntityManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Transformable;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vital.VitalEntryFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObject;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;

@Repository
public class VitalDao extends AbstractDao {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(VitalDao.class);

    private static final String WHERE_ID = "from VitalEntryPo where id = :id and userId = :patientId ";
    private static final String WHERE_PATIENT_ID = "from VitalEntryPo where userId = :patientId ";

    public VitalEntries get(String patientId) {
        return get(patientId, new VitalEntryFilter());
    }

    public VitalEntries get(String patientId, VitalEntryFilter filter) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving Vitals info for patient");

        VitalsJpaFilter vitalsJpaFilter = new VitalsJpaFilter(filter, "entryDate");

        SimpleEntityManager<VitalEntryPo> simpleEntityManager = new SimpleEntityManager<VitalEntryPo>(WHERE_PATIENT_ID, entityManager);

        List<VitalEntryPo> results = simpleEntityManager.get(patientId, vitalsJpaFilter, VitalEntryPo.class);

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        VitalEntries vitals = new VitalEntries();
        Mapper.add(transformableList, vitals);

        return vitals;
    }

    private List<Transformable<? extends DomainTransferObject>> createTransformableList(List<VitalEntryPo> results) {
        List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
        transformableList.addAll(results);

        return transformableList;
    }

    public VitalEntry get(String patientId, String vitalEntryId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving Vital info for event");

        SimpleEntityManager<VitalEntryPo> simpleEntityManager = new SimpleEntityManager<VitalEntryPo>(WHERE_ID, entityManager);
        List<VitalEntryPo> results = simpleEntityManager.get(patientId, vitalEntryId, VitalEntryPo.class);

        if (results.size() == 0)
            return null;

        if (results.size() > 1)
            logger.warn("Found more than one Vital record");

        return results.get(0).create();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public VitalEntry save(VitalEntry entry) {
        if (logger.isDebugEnabled()) logger.debug("Saving Vital");

        UserHistoryEntityManager<VitalEntryPo> simpleEntityManager = new UserHistoryEntityManager<VitalEntryPo>(entityManager, VitalEntryPo.TABLE_NAME, entry);
        VitalEntryPo po = simpleEntityManager.save(new VitalEntryPo(entry));

        if (logger.isDebugEnabled()) logger.debug("Saved Vital");

        return po.create();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void delete(VitalEntry entry) {
        if (logger.isDebugEnabled()) logger.debug("Deleting Vital");

        UserHistoryEntityManager<VitalEntryPo> simpleEntityManager = new UserHistoryEntityManager<VitalEntryPo>(entityManager, VitalEntryPo.TABLE_NAME);
        simpleEntityManager.delete(entry, VitalEntryPo.class);
    }

}
