package com.agilex.healthcare.mobilehealthplatform.datalayer.paingoal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Mapper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Transformable;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObject;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoal;
import com.agilex.healthcare.mobilehealthplatform.domain.PainGoals;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PainGoalStatus;

@Repository
public class PainGoalDao extends AbstractDao {

	public PainGoals getPainGoals(String patientId) {
		TypedQuery<PainGoalPo> constructedQuery = this.constructWhereClause(patientId);

		PainGoals goals = executeAndFetchGoals(constructedQuery);
 
		return goals;
	}

	public PainGoal getPainGoal(String patientId, String dataId) {
		TypedQuery<PainGoalPo> constructedQuery = this.constructWhereClause(patientId, dataId);
		PainGoals goals = executeAndFetchGoals(constructedQuery);
		
		if(goals.size() == 0){
			return null;
		}
		
		return goals.get(0);
	}

	public PainGoals getAllPainGoals(String patientId) {
		TypedQuery<PainGoalPo> constructedQuery = this.constructWhereClauseForAllGoals(patientId);

		PainGoals goals = executeAndFetchGoals(constructedQuery);

		return goals;
	}

	private PainGoals executeAndFetchGoals(TypedQuery<PainGoalPo> constructedQuery) {
		List<PainGoalPo> goalsPo = this.executeForPainGoals(constructedQuery);

		List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(goalsPo);
		PainGoals goals = new PainGoals();
		Mapper.add(transformableList, goals);
		return goals;
	}

	private List<PainGoalPo> executeForPainGoals(
			TypedQuery<PainGoalPo> constructedQuery) {
		return constructedQuery.getResultList();
	}

	private List<Transformable<? extends DomainTransferObject>> createTransformableList(
			List<PainGoalPo> goalsPo) {
		List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
		transformableList.addAll(goalsPo);

		return transformableList;
	}

	private TypedQuery<PainGoalPo> constructWhereClause(String patientId) {
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("patientId", patientId);
		parameters.put("deletedStatus",
				PainGoalStatus.PainGoalStatusDeleted.toString());

		String query = "from PainGoalPo goal where goal.userId = :patientId and goal.status != :deletedStatus";

		TypedQuery<PainGoalPo> constructedQuery = this.setParametersForQuery(
				query, parameters);
		return constructedQuery;
	}

	private TypedQuery<PainGoalPo> constructWhereClause(String patientId, String painGoalId) {
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("patientId", patientId);
		parameters.put("painGoalId", painGoalId);
		parameters.put("deletedStatus",
				PainGoalStatus.PainGoalStatusDeleted.toString());

		String query = "from PainGoalPo goal where goal.userId = :patientId and goal.id = :painGoalId and goal.status != :deletedStatus";

		TypedQuery<PainGoalPo> constructedQuery = this.setParametersForQuery(
				query, parameters);
		return constructedQuery;
	}

	private TypedQuery<PainGoalPo> constructWhereClauseForAllGoals(
			String patientId) {
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("patientId", patientId);

		String query = "from PainGoalPo goal where goal.userId = :patientId";

		TypedQuery<PainGoalPo> constructedQuery = this.setParametersForQuery(
				query, parameters);
		return constructedQuery;
	}

	private TypedQuery<PainGoalPo> setParametersForQuery(String query,
			Map<String, Object> parameters) {
		TypedQuery<PainGoalPo> constructedQuery = this.entityManager
				.createQuery(query, PainGoalPo.class);
		for (String key : parameters.keySet()) {
			constructedQuery.setParameter(key, parameters.get(key));
		}
		return constructedQuery;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public PainGoal createOrUpdateNewPainGoal(PainGoal data) {
		PainGoalPo goalPoToSave = new PainGoalPo(data);

		if (NullChecker.isNotNullish(goalPoToSave.getId())) {
			PainGoalPo existingPo = this.entityManager.find(PainGoalPo.class,
					goalPoToSave.getId());
			if (existingPo != null && existingPo.getPainGoalEntries() != null) {
				goalPoToSave
						.setPainGoalEntries(existingPo.getPainGoalEntries());

				for (PainGoalEntryPo entryPo : goalPoToSave
						.getPainGoalEntries()) {
					if (entryPo.getStatus() == null) {
						entryPo.setStatus("Active");
					}
				}

			}
		}

		UserHistoryEntityManager<PainGoalPo> simpleEntityManager = new UserHistoryEntityManager<PainGoalPo>(
				entityManager, PainGoalPo.TABLE_NAME, data);
		PainGoalPo po = simpleEntityManager.save(goalPoToSave);

		return po.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deletePainGoal(PainGoal data) {
		UserHistoryEntityManager<PainGoalPo> simpleEntityManager = new UserHistoryEntityManager<PainGoalPo>(
				entityManager, PainGoalPo.TABLE_NAME);
		simpleEntityManager.delete(data, PainGoalPo.class);
	}

}
