package com.agilex.healthcare.mobilehealthplatform.datalayer.paindiary;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;

import com.agilex.healthcare.mobilehealthplatform.datalayer.Transformable;
import com.agilex.healthcare.mobilehealthplatform.datalayer.PersistenceObject;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamImplicit;

import org.hibernate.annotations.GenericGenerator;

import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryQuestionId;
import com.agilex.healthcare.mobilehealthplatform.domain.PainDiaryReport;
import com.agilex.healthcare.utility.NullChecker;

@Entity
@Table(name = PainDiaryEntryPo.TABLE_NAME)
@XStreamAlias("PainDiaryEntry")
public class PainDiaryEntryPo extends PersistenceObject implements Transformable<PainDiaryEntry> {

	private static final long serialVersionUID = -2589048511612117409L;
    public static final String TABLE_NAME = "ENTRIES";

    private Date dateCompleted;	
	private boolean isCompleted;
	private Date dateCreated;
    
	@XStreamImplicit(itemFieldName="PainResponses")
	private List<PainDiaryResponsePo> painResponses;
	
	public PainDiaryEntryPo() {}
	
	public PainDiaryEntryPo(PainDiaryEntry dto) {
		setId(dto.getUniqueId());
		setUserId(dto.getPatientId());

		isCompleted = dto.isCompleted();
		dateCompleted = dto.getDateCompleted();
		dateCreated = dto.getDateCreated();

        createResponseList(dto.getPainQuestionValues());
    }
	
	private void createResponseList(PainDiaryReport painQuestionValues) {
        painResponses = new ArrayList<PainDiaryResponsePo>();

        createAddResponse(PainDiaryQuestionId.PAIN_TYPE, painQuestionValues.getPainType());
        createAddResponse(PainDiaryQuestionId.ITEMS_THAT_ACTIVATED_PAIN, painQuestionValues.getItemsThatActivatedPain());

        if (painQuestionValues.getPainQuality() != null)
            for (String value : painQuestionValues.getPainQuality())
                createAddResponse(PainDiaryQuestionId.PAIN_QUALITY, value);

        if (painQuestionValues.getTreatment() != null)
            for (String value : painQuestionValues.getTreatment())
                createAddResponse(PainDiaryQuestionId.TREATMENT, value);

        createAddResponse(PainDiaryQuestionId.USEFULNESS_OF_TREATMENT, painQuestionValues.getUsefulnessOfTreatment());
        createAddResponse(PainDiaryQuestionId.AVERAGE_PAIN_INTENSITY, painQuestionValues.getAveragePainIntensity());
        createAddResponse(PainDiaryQuestionId.PAIN_INTERFERENCE, painQuestionValues.getPainInterference());
        createAddResponse(PainDiaryQuestionId.MOOD_LEVEL, painQuestionValues.getMoodLevel());
        createAddResponse(PainDiaryQuestionId.AMOUNT_REFRESHED_AFTER_SLEEP, painQuestionValues.getAmountRefreshedAfterSleep());
        createAddResponse(PainDiaryQuestionId.DEPRESSION_LEVEL, painQuestionValues.getDepressionLevel());
        createAddResponse(PainDiaryQuestionId.IMPACT_ON_WALKING, painQuestionValues.getImpactOnWalking());
        createAddResponse(PainDiaryQuestionId.ITEMS_THAT_WORSEN_PAIN, painQuestionValues.getItemsThatWorsenPain());
        createAddResponse(PainDiaryQuestionId.ITEMS_THAT_IMPROVED_PAIN, painQuestionValues.getItemsThatImprovedPain());
        createAddResponse(PainDiaryQuestionId.COMMENTS, painQuestionValues.getComments());
	}

    private void createAddResponse(String questionId, String value) {
        if (NullChecker.isNullish(value))
            return;

        PainDiaryResponsePo response = new PainDiaryResponsePo();
        response.setEntry(this);
        response.setQuestionId(questionId);
        response.setResponse(value);

        painResponses.add(response);
    }

	@Id
	@GeneratedValue(generator = "pain_diary_entry_id_generator")
	@GenericGenerator(name = "pain_diary_entry_id_generator", strategy = "uuid")
	@Column(name = "ENTRY_ID", nullable = false)
	public String getId() {
		return super.getId();
	}

	@Column(name = "DATE_COMPLETED")
	public Date getDateCompleted() {
		return dateCompleted;
	}
	public void setDateCompleted(Date dateCompleted) {
		this.dateCompleted = dateCompleted;
	}
	
	@OneToMany(cascade = CascadeType.ALL, fetch = FetchType.EAGER, mappedBy = "entry")
	public List<PainDiaryResponsePo> getPainResponses(){
		return painResponses;
	}
	public void setPainResponses(List<PainDiaryResponsePo> painResponses) {
		this.painResponses = painResponses;
	}
	
	@Column(name = "COMPLETED")
	public boolean isCompleted() {
		return isCompleted;
	}
	public void setCompleted(boolean isCompleted) {
		this.isCompleted = isCompleted;
	}

	@Column(name = "PATIENT_ID")
	public String getUserId() {
		return super.getUserId();
	}

	@Column(name = "DATE_CREATED")
	public Date getDateCreated() {
		return dateCreated;
	}

	public void setDateCreated(Date dateCreated) {
		this.dateCreated = dateCreated;
	}

	@Override
	public PainDiaryEntry create() {
		PainDiaryEntry dto = new PainDiaryEntry();
		

		dto.setCompleted(isCompleted);
		dto.setUniqueId(getId());
		dto.setDateCompleted(dateCompleted);
		dto.setPatientId(getUserId());
		dto.setDateCreated(dateCreated);
		
		dto.setPainQuestionValues(createResponseReport());
		
		return dto;
	}
	
	private PainDiaryReport createResponseReport() {
		PainDiaryReport report = new PainDiaryReport();
	
		if (painResponses != null) {
			for (PainDiaryResponsePo response : painResponses) {
				addResponseToReport(response, report);
			}
		}
		
		return report;
	}

	private void addResponseToReport(PainDiaryResponsePo response, PainDiaryReport report) {
		String questionId = response.getQuestionId();

		if (PainDiaryQuestionId.AMOUNT_REFRESHED_AFTER_SLEEP.equalsIgnoreCase(questionId)) {
			report.setAmountRefreshedAfterSleep(response.getResponse());
		} else if (PainDiaryQuestionId.AVERAGE_PAIN_INTENSITY.equalsIgnoreCase(questionId)) {
			report.setAveragePainIntensity(response.getResponse());
		} else if (PainDiaryQuestionId.COMMENTS.equalsIgnoreCase(questionId)) {
			report.setComments(response.getResponse());
		} else if (PainDiaryQuestionId.DEPRESSION_LEVEL.equalsIgnoreCase(questionId)) {
			report.setDepressionLevel(response.getResponse());
		} else if (PainDiaryQuestionId.IMPACT_ON_WALKING.equalsIgnoreCase(questionId)) {
			report.setImpactOnWalking(response.getResponse());
		} else if (PainDiaryQuestionId.ITEMS_THAT_ACTIVATED_PAIN.equalsIgnoreCase(questionId)) {
			report.setItemsThatActivatedPain(response.getResponse());
		} else if (PainDiaryQuestionId.ITEMS_THAT_IMPROVED_PAIN.equalsIgnoreCase(questionId)) {
			report.setItemsThatImprovedPain(response.getResponse());
		} else if (PainDiaryQuestionId.ITEMS_THAT_WORSEN_PAIN.equalsIgnoreCase(questionId)) {
			report.setItemsThatWorsenPain(response.getResponse());
		} else if (PainDiaryQuestionId.MOOD_LEVEL.equalsIgnoreCase(questionId)) {
			report.setMoodLevel(response.getResponse());
		} else if (PainDiaryQuestionId.PAIN_INTERFERENCE.equalsIgnoreCase(questionId)) {
			report.setPainInterference(response.getResponse());
		} else if (PainDiaryQuestionId.PAIN_QUALITY.equalsIgnoreCase(questionId)) {
			if (NullChecker.isNotNullish(report.getPainQuality())) {
				report.getPainQuality().add(response.getResponse());
			} else { 
				List<String> values = new ArrayList<String>();
				values.add(response.getResponse());
				report.setPainQuality(values);
			}
		} else if (PainDiaryQuestionId.PAIN_TYPE.equalsIgnoreCase(questionId)) {
			report.setPainType(response.getResponse());
		} else if (PainDiaryQuestionId.TREATMENT.equalsIgnoreCase(questionId)) {
			if (NullChecker.isNotNullish(report.getTreatment())) {
				report.getTreatment().add(response.getResponse());
			} else {
				List<String> values = new ArrayList<String>();
				values.add(response.getResponse());
				report.setTreatment(values);
			}
		} else if (PainDiaryQuestionId.USEFULNESS_OF_TREATMENT.equalsIgnoreCase(questionId)) {
			report.setUsefulnessOfTreatment(response.getResponse());
		} 
	}
	
	
	
	
}
