package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.Query;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.PersistenceObject;
import com.agilex.healthcare.mobilehealthplatform.datalayer.validPo;

@Repository
public class NotificationDao extends AbstractDao {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(NotificationDao.class);
	private static final int MAX_NUMBER_OF_RETURNED_NOTIFICATIONS = 50;

	public List<NotificationPo> getActiveNotifications(String patientId, DateFilter filter) {
		Query constructedQuery = this.constructWhereClause(patientId, false);
		List<NotificationPo> notifications = this.executeForNotifications(constructedQuery);
		return DateFilterer.filterByDate(notifications, filter);
	}

	public NotificationPo getActiveNotification(String userId, String notificationId) {
		String selectSingleNotification = "from NotificationPo where userId = :userId and id = :notificationId and activeFlag = true ";
		Query query = getQuery(selectSingleNotification);
		query.setParameter("notificationId", notificationId);
		query.setParameter("userId", userId);

		return this.executeForNotification(query);
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public NotificationPo updateNotification(NotificationPo notification) {
		validPo<NotificationPo> validPo = new validPo<NotificationPo>(entityManager, notification);
		if (validPo.isOk()) {
			return entityManager.merge(notification);
		} else {
			return null;
		}
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public NotificationPo saveNotification(NotificationPo notification) {
		validPo<NotificationPo> validPo = new validPo<NotificationPo>(entityManager, notification);
		if (validPo.isOk()) {
			return entityManager.merge(notification);
		}
		return null;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deleteNotification(NotificationPo notification) {
		validPo<NotificationPo> validPo = new validPo<NotificationPo>(entityManager, notification);
		if (validPo.isOk()) {
			NotificationPo po = validPo.getPo();
			if (po != null) {
				po.setDeletedDate(new Date());
				po.setActiveFlag(false);
				entityManager.merge(po);
			}
		}
	}

	private Query constructWhereClause(String userId, boolean todayFlag) {
		StringBuilder whereClause = new StringBuilder();

		Map<String, Object> parameters = new HashMap<String, Object>();

		whereClause.append(" where userId = :userId and activeFlag = :activeFlag ");
		if (todayFlag) {
			whereClause.append("and n.date >= :today and n.date <= :tomorrow ");
			DateTime date = new DateTime().toDateMidnight().toDateTime();
			DateTime tomorrow = date.plusDays(1);
			parameters.put("today", date.toDate());
			parameters.put("tomorrow", tomorrow.toDate());
		}
		whereClause.append("order by n.date DESC");

		parameters.put("userId", userId);
		parameters.put("activeFlag", true);

		String query = "from NotificationPo n" + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private Query setParametersForQuery(String query, Map<String, Object> parameters) {
		//LOGGER.debug(String.format(":::: Query: %s", query));
		Query constructedQuery = this.entityManager.createQuery(query);
		for (String key : parameters.keySet()) {
			constructedQuery.setParameter(key, parameters.get(key));
		}
		return constructedQuery;
	}

	private List<NotificationPo> executeForNotifications(Query query) {
		@SuppressWarnings("unchecked")
		List<NotificationPo> notificationResults = query.getResultList();
		int endIndex = MAX_NUMBER_OF_RETURNED_NOTIFICATIONS > notificationResults.size() ?  notificationResults.size() : MAX_NUMBER_OF_RETURNED_NOTIFICATIONS;
		List<NotificationPo> notifications = new ArrayList<NotificationPo>();
		notifications.addAll(notificationResults.subList(0, endIndex));
		return notifications;
	}

	private NotificationPo executeForNotification(Query query) {
		try {
			return (NotificationPo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private Query getQuery(String dbQuery) {
		Query query = this.entityManager.createQuery(dbQuery);
		return query;
	}
}
