package com.agilex.healthcare.mobilehealthplatform.datalayer.notification;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.validPo;
import com.agilex.healthcare.utility.NullChecker;

@Repository
public class DeviceRegistrationDao extends AbstractDao {
	
	public List<DeviceRegistrationPo> fetchDeviceRegistrations(String userId) {
		List<String> userIds = new ArrayList<String>();
		userIds.add(userId);
		return fetchDeviceRegistrations(userIds);
	}

	public List<DeviceRegistrationPo> fetchDeviceRegistrations(List<String> userIds) {
		String selectForMultipleUsersQueryString = "from DeviceRegistrationPo po where po.userId IN (:userIds) and po.optedIn = true";
		TypedQuery<DeviceRegistrationPo> query = getQuery(selectForMultipleUsersQueryString);
		query.setParameter("userIds", userIds);

		return this.executeForDeviceRegistrations(query);
	}

	public DeviceRegistrationPo fetchDeviceRegistrationById(String deviceRegistrationId) {
		String selectSingleDeviceRegistration = "from DeviceRegistrationPo po where po.id = :deviceRegistrationId";
		TypedQuery<DeviceRegistrationPo> query = getQuery(selectSingleDeviceRegistration);
		query.setParameter("deviceRegistrationId", deviceRegistrationId);

		return this.executeForDeviceRegistration(query);
	}

	public DeviceRegistrationPo fetchDeviceRegistration(String userId, String deviceToken) {
		String selectSingleDeviceRegistration = "from DeviceRegistrationPo po where po.userId = :userId and po.deviceToken = :deviceToken";
		TypedQuery<DeviceRegistrationPo> query = entityManager.createQuery(selectSingleDeviceRegistration, DeviceRegistrationPo.class);
		query.setParameter("userId", userId);
		query.setParameter("deviceToken", deviceToken);

		return this.executeForDeviceRegistration(query);
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public DeviceRegistrationPo saveDeviceRegistration(DeviceRegistrationPo deviceRegistration) {
		validPo<DeviceRegistrationPo> validPo = new validPo<DeviceRegistrationPo>(entityManager, deviceRegistration);
		if (validPo.isOk()) {
			return entityManager.merge(deviceRegistration);
		}
		return null;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deleteDeviceRegistration(DeviceRegistrationPo deviceRegistration) {
		validPo<DeviceRegistrationPo> validPo = new validPo<DeviceRegistrationPo>(entityManager, deviceRegistration);
		if (validPo.isOk() && validPo.getPo() != null) {
			entityManager.remove(validPo.getPo());
		}
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deleteDeviceRegistrationByToken(String deviceToken) {
		String deleteByToken = "delete from DeviceRegistrationPo po where po.deviceToken = :deviceToken";
		Query query = entityManager.createQuery(deleteByToken);
		query.setParameter("deviceToken", deviceToken);
		
		query.executeUpdate();
	}
	
	@Transactional(propagation = Propagation.REQUIRED)
	public void updateOptStatus(String userId, boolean status) {
		Query updateRegistrations = entityManager.createNativeQuery("UPDATE DEVICE_REGISTRATION SET OPTED_IN_FLAG = :status WHERE USER_ID = :userId", DeviceRegistrationPo.class);
		updateRegistrations.setParameter("userId", userId);
		updateRegistrations.setParameter("status", status);
		
		updateRegistrations.executeUpdate();
	}

	private List<DeviceRegistrationPo> executeForDeviceRegistrations(TypedQuery<DeviceRegistrationPo> query) {
		List<DeviceRegistrationPo> deviceRegistrationResults = query.getResultList();
		return deviceRegistrationResults;
	}

	private DeviceRegistrationPo executeForDeviceRegistration(TypedQuery<DeviceRegistrationPo> query) {
		List<DeviceRegistrationPo> deviceRegistrations = query.getResultList();
		if(NullChecker.isNotNullish(deviceRegistrations)) {
			return deviceRegistrations.get(0);
		}
		return null;
	}

	private TypedQuery<DeviceRegistrationPo> getQuery(String dbQuery) {
		TypedQuery<DeviceRegistrationPo> query = this.entityManager.createQuery(dbQuery, DeviceRegistrationPo.class);
		return query;
	}

}
