package com.agilex.healthcare.mobilehealthplatform.datalayer.mygoals;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.Query;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Mapper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.SimpleEntityManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Transformable;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObject;
import com.agilex.healthcare.mobilehealthplatform.domain.Mygoals;
import com.agilex.healthcare.mobilehealthplatform.domain.MygoalsList;

@Repository
public class MygoalsDao extends AbstractDao {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MygoalsDao.class);

    
    public static final String WHERE_MYGOALS = "select a from MygoalsPo a where a.userId = :patientId and a.inventoryType = :inventoryType and " +
			 									"a.dateUpdated = (select max(po.dateUpdated) from MygoalsPo po where po.userId = :patientId and po.inventoryType = :inventoryType)";

    public static final String WHERE_LATEST_MYGOALS = "SELECT a.* FROM MYGOALS a "+ 
																	    "INNER JOIN "+
																	        "(SELECT QUESTION_KEY, MAX(DATE_UPDATED) AS MAX_DATE FROM MYGOALS WHERE PATIENT_ID = :patientId GROUP BY QUESTION_KEY ) b " +
																	     "ON a.QUESTION_KEY = b.QUESTION_KEY AND a.DATE_UPDATED = b.MAX_DATE ORDER BY INVENTORY_TYPE, DATE_UPDATED DESC";

    public static final String WHERE_ID = "from MygoalsPo where userId = :patientId and id = :id";
    public static final String WHERE_PATIENT_ID = "from MygoalsPo where userId = :patientId";
    
    public Mygoals get(String patientId, String invRespId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving mygoals");
        Query query = getQuery(WHERE_ID);
        query.setParameter("patientId", patientId);
 		query.setParameter("id", invRespId);
 		MygoalsPo mygoalsPo = executeForMygoals(query);
 		return mygoalsPo.create();
    }
	
    public MygoalsList getLatestMygoalsListByType(String patientId, String invType) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving mygoals for patient and inventory type");
		
		MygoalsJpaFilter filter = new MygoalsJpaFilter(patientId, invType);
        SimpleEntityManager<MygoalsPo> simpleEntityManager = new SimpleEntityManager<MygoalsPo>(WHERE_MYGOALS, entityManager);
        List<MygoalsPo> results = simpleEntityManager.get(filter, MygoalsPo.class);

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        MygoalsList inventoryResponses = new MygoalsList();
        Mapper.add(transformableList, inventoryResponses);

        return inventoryResponses;
    }

    public MygoalsList getLatestMygoalsList(String patientId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving mygoalsList for patient");

        Query query = entityManager.createNativeQuery(WHERE_LATEST_MYGOALS, MygoalsPo.class);
        query.setParameter("patientId", patientId);
        List<MygoalsPo> results = query.getResultList();

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        MygoalsList mygoalsList = new MygoalsList();
        Mapper.add(transformableList, mygoalsList);

		return mygoalsList;
    }
 

    public MygoalsList getMygoalsListByPatientId(String patientId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving mygoals for patient");

        Query query = entityManager.createQuery(WHERE_PATIENT_ID, MygoalsPo.class);
        query.setParameter("patientId", patientId);
        List<MygoalsPo> results = query.getResultList();

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        MygoalsList inventoryResponses = new MygoalsList();
        Mapper.add(transformableList, inventoryResponses);

        return inventoryResponses;
        
    }
 
    @Transactional(propagation = Propagation.REQUIRED)
    public Mygoals save(Mygoals invResp) {
        if (logger.isDebugEnabled()) logger.debug("Saving mygoals");

        removePotentialExistingDraft(invResp);

        UserHistoryEntityManager<MygoalsPo> simpleEntityManager = new UserHistoryEntityManager<MygoalsPo>(entityManager, MygoalsPo.TABLE_NAME, invResp);
        MygoalsPo po = simpleEntityManager.save(new MygoalsPo(invResp));

        if (logger.isDebugEnabled()) logger.debug("Saved mygoals");

        return po.create();
    }
   
    private List<Transformable<? extends DomainTransferObject>> createTransformableList(List<MygoalsPo> results) {
        List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
        transformableList.addAll(results);

        return transformableList;
    }
    
	private Query getQuery(String dbQuery) {
		Query query = this.entityManager.createQuery(dbQuery);
		return query;
	}
	
	private MygoalsPo executeForMygoals(Query query) {
		try {
			return (MygoalsPo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}
	

    private void removePotentialExistingDraft(Mygoals mygoals) {
        entityManager.createQuery("delete MygoalsPo where userId = :patientId and id = :id")
                .setParameter("patientId", mygoals.getPatientId())
                .setParameter("id", mygoals.getId())
                .executeUpdate();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void delete(Mygoals invResp) {
        if (logger.isDebugEnabled()) 
        		logger.debug("Deleting mygoals");

        invResp.setUniqueId(invResp.getId());
        UserHistoryEntityManager<MygoalsPo> simpleEntityManager = new UserHistoryEntityManager<MygoalsPo>(entityManager, MygoalsPo.TABLE_NAME);
        simpleEntityManager.delete(invResp, MygoalsPo.class);
    }
}