package com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill;

import java.util.*;

import javax.persistence.TypedQuery;

import org.springframework.stereotype.*;
import org.springframework.transaction.annotation.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillRequestDataLayer;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.utility.NullChecker;

@Service
@Repository
public class MedicationRefillRequestDataLayerRepo extends AbstractDao implements MedicationRefillRequestDataLayer {

//	private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(MedicationRefillRequestDataLayerRepo.class);

	private static final String WHERE_IDs = "from MedicationRefillRequestPo where medicationId = :medicationId and patientId = :patientId order by requestDate desc";
	private static final String WHERE_IDs_IN = "from MedicationRefillRequestPo where medicationId  in (:medicationIds) and patientId = :patientId order by requestDate desc";
	private static final String WHERE_PATIENT_ID = "from MedicationRefillRequestPo where patientId = :patientId";

	@Transactional(propagation = Propagation.REQUIRED)
	public MedicationRefillRequest saveRefillRequest(MedicationRefillRequest refillRequest) {
		UserHistoryEntityManager<MedicationRefillRequestPo> userHistoryEntityManager = new UserHistoryEntityManager<MedicationRefillRequestPo>(entityManager, MedicationRefillRequestPo.TABLE_NAME, refillRequest);
		MedicationRefillRequestPo po = userHistoryEntityManager.save(new MedicationRefillRequestPo(refillRequest));

		return po.create();
	}

	public MedicationRefillRequest getRefillRequest(PatientIdentifier patientIdentifier, String medicationId) {
		TypedQuery<MedicationRefillRequestPo> byIdsQuery = this.entityManager.createQuery(WHERE_IDs, MedicationRefillRequestPo.class);
		byIdsQuery.setParameter("medicationId", medicationId);
		byIdsQuery.setParameter("patientId", patientIdentifier.getUniqueId());
		List<MedicationRefillRequestPo> results = byIdsQuery.getResultList();
		
		if (results == null || results.size() == 0)
			return null;

		return results.get(0).create();
	}
	
	public MedicationRefillRequests fetchLatestRefillRequests(PatientIdentifier patientIdentifier, List<String> medicationIds) {
		if(NullChecker.isNullish(medicationIds))
			return new MedicationRefillRequests();
		
		TypedQuery<MedicationRefillRequestPo> byIdsQuery = this.entityManager.createQuery(WHERE_IDs_IN, MedicationRefillRequestPo.class);
		byIdsQuery.setParameter("medicationIds", medicationIds);
		byIdsQuery.setParameter("patientId", patientIdentifier.getUniqueId());
		List<MedicationRefillRequestPo> results = byIdsQuery.getResultList();
		
		List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);
		
		MedicationRefillRequests requests = new MedicationRefillRequests();
		Mapper.add(transformableList, requests);
		
		return filterDuplicateIds(requests);
	}

	public MedicationRefillRequests getRefillRequests(PatientIdentifier patientIdentifier, DateFilter dateFilter) {
		JpaFilter jpaFilter = new JpaDateFilter(dateFilter, "requestDate");
		SimpleEntityManager<MedicationRefillRequestPo> simpleEntityManager = new SimpleEntityManager<MedicationRefillRequestPo>(WHERE_PATIENT_ID, entityManager);

		List<MedicationRefillRequestPo> results = simpleEntityManager.get(patientIdentifier.getUniqueId(), jpaFilter, MedicationRefillRequestPo.class);

		List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

		MedicationRefillRequests requests = new MedicationRefillRequests();
		Mapper.add(transformableList, requests);

		return DateFilterer.filterByDate(requests, dateFilter);
	}

	private List<Transformable<? extends DomainTransferObject>> createTransformableList(List<MedicationRefillRequestPo> results) {
		List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
		transformableList.addAll(results);

		return transformableList;
	}
	
	MedicationRefillRequests filterDuplicateIds(MedicationRefillRequests refillRequests) {
		Map<String, MedicationRefillRequest> requestMapping = new HashMap<String, MedicationRefillRequest>();
		
		MedicationRefillRequests filtered = new MedicationRefillRequests();
		for(MedicationRefillRequest refillRequest : refillRequests) {
			MedicationRefillRequest medicationRefillRequest = requestMapping.get(refillRequest.getMedicationId());
			if(medicationRefillRequest == null)
				requestMapping.put(refillRequest.getMedicationId(), refillRequest);
		}
		filtered.addAll(requestMapping.values());
		
		return filtered;
	}

}
