package com.agilex.healthcare.mobilehealthplatform.datalayer.healthadvocaterequest;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Query;
import javax.persistence.TypedQuery;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.domain.HealthAdvocateStatusCode;
import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.healthadvocaterequest.HealthAdvocateRequestPo;
import com.agilex.healthcare.utility.DateHelper;

@Repository
public class HealthAdvocateRequestDao extends AbstractDao {

	public List<HealthAdvocateRequestPo> getHealthAdvocateRequests(String patientId) {
		TypedQuery<HealthAdvocateRequestPo> constructedQuery = this.constructWhereClause(patientId);
		return this.executeForHealthAdvocateRequests(constructedQuery);
	}

	private TypedQuery<HealthAdvocateRequestPo> constructWhereClause(String patientId) {
		StringBuilder whereClause = new StringBuilder();
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("patientId", patientId);
		
		Date today = new Date();
		
		whereClause.append(" and healthAdvocateRequest.status IS NOT NULL");
		whereClause.append(" and healthAdvocateRequest.lastActionDate IS NOT NULL");
		
		whereClause.append(" and ((healthAdvocateRequest.status = :submittedStatus OR healthAdvocateRequest.status = :acceptedStatus) OR healthAdvocateRequest.lastActionDate >= :oldestPossibleDate)");

		parameters.put("submittedStatus", HealthAdvocateStatusCode.SUBMITTED);
		parameters.put("acceptedStatus", HealthAdvocateStatusCode.ACCEPTED);
		parameters.put("oldestPossibleDate", DateHelper.beginOfDate(DateHelper.minusDays(today, 180)));

		
		String query = "from HealthAdvocateRequestPo healthAdvocateRequest where healthAdvocateRequest.patientEDIPI = :patientId " + whereClause.toString();
		TypedQuery<HealthAdvocateRequestPo> constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private TypedQuery<HealthAdvocateRequestPo> setParametersForQuery(String query, Map<String, Object> parameters) {
		TypedQuery<HealthAdvocateRequestPo> constructedQuery = this.entityManager.createQuery(query, HealthAdvocateRequestPo.class);
		for (String key : parameters.keySet()) {
			constructedQuery.setParameter(key, parameters.get(key));
		}
		return constructedQuery;
	}

	private List<HealthAdvocateRequestPo> executeForHealthAdvocateRequests(TypedQuery<HealthAdvocateRequestPo> constructedQuery) {
		return constructedQuery.getResultList();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public HealthAdvocateRequestPo createOrUpdateNewHealthAdvocateRequest(HealthAdvocateRequestPo data) {
		HealthAdvocateRequestPo healthAdvocateRequestPo = this.entityManager.merge(data);
		return healthAdvocateRequestPo;
	}

	public HealthAdvocateRequestPo getHealthAdvocateRequest(String patientId, String dataId) {
		String queryStringForId = "from HealthAdvocateRequestPo healthAdvocateRequest where healthAdvocateRequest.patientEDIPI = :patientId and healthAdvocateRequest.id = :dataId";
		Map<String, Object> parameters = new HashMap<String, Object>();
		parameters.put("patientId", patientId);
		parameters.put("dataId", dataId);
		
		TypedQuery<HealthAdvocateRequestPo> constructedQuery = this.setParametersForQuery(queryStringForId, parameters);
		
		return this.executeForHealthAdvocateRequest(constructedQuery);
	}
	
	@Transactional(propagation = Propagation.REQUIRED)
	public HealthAdvocateRequestFormPo saveHealthAdvocateRequestForm(HealthAdvocateRequestFormPo form) {
		HealthAdvocateRequestFormPo result = this.entityManager.merge(form);
		return result;
	}
	
	public HealthAdvocateRequestFormPo getHealthAdvocateRequestFormPo(HealthAdvocateRequestPo request) {
		TypedQuery<HealthAdvocateRequestFormPo> query = this.entityManager.createQuery("from HealthAdvocateRequestFormPo form where form.requestId = :requestId and form.status = :status", HealthAdvocateRequestFormPo.class);
		query.setParameter("requestId", request.getId());
		query.setParameter("status", HealthAdvocateStatusCode.SUBMITTED);
		
		List<HealthAdvocateRequestFormPo> result = query.getResultList();
		
		if(result == null || result.size() == 0)
			return null;
		
		return result.get(0);
	}

	private HealthAdvocateRequestPo executeForHealthAdvocateRequest(TypedQuery<HealthAdvocateRequestPo> constructedQuery) {

		List<HealthAdvocateRequestPo> results = constructedQuery.getResultList();
		if (results.size() > 0) {
			return results.get(0);
		} else {
			return null;
		}

	}
	
	
}
