package com.agilex.healthcare.mobilehealthplatform.datalayer.exercise;

import java.util.ArrayList;
import java.util.List;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.JpaDateFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.JpaFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Mapper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.SimpleEntityManager;
import com.agilex.healthcare.mobilehealthplatform.datalayer.Transformable;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObject;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercise;
import com.agilex.healthcare.mobilehealthplatform.domain.Exercises;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;

@Repository
public class ExerciseDao extends AbstractDao {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(ExerciseDao.class);

    private static final String WHERE_ID = "from ExercisePo where id = :id and userId = :patientId";
    private static final String WHERE_PATIENT_ID = "from ExercisePo where userId = :patientId ";

    public Exercises get(String patientId) {
        return get(patientId, DateFilterFactory.createEmptyFilter());
    }

    public Exercises get(String patientId, DateFilter filter) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving Exercise info for patient");

        JpaFilter jpaFilter = new JpaDateFilter(filter, "activityDate");
        SimpleEntityManager<ExercisePo> simpleEntityManager = new SimpleEntityManager<ExercisePo>(WHERE_PATIENT_ID, entityManager);

        List<ExercisePo> results = simpleEntityManager.get(patientId, jpaFilter, ExercisePo.class);

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        Exercises exercises = new Exercises();
        Mapper.add(transformableList, exercises);

        return exercises;
    }

    private List<Transformable<? extends DomainTransferObject>> createTransformableList(List<ExercisePo> results) {
         List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
        transformableList.addAll(results);

        return transformableList;
    }

    public Exercise get(String patientId, String eventId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving Exercise info for an event");

        SimpleEntityManager<ExercisePo> simpleEntityManager = new SimpleEntityManager<ExercisePo>(WHERE_ID, entityManager);
        List<ExercisePo> results = simpleEntityManager.get(patientId, eventId, ExercisePo.class);

        if (results.size() == 0)
            return null;

        if (results.size() > 1)
            logger.warn("Found more than one Exercise record");

        return results.get(0).create();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Exercise save(Exercise exercise) {
        if (logger.isDebugEnabled()) logger.debug("Saving Exercise");

        UserHistoryEntityManager<ExercisePo> simpleEntityManager = new UserHistoryEntityManager<ExercisePo>(entityManager, ExercisePo.TABLE_NAME, exercise);
        ExercisePo po = simpleEntityManager.save(new ExercisePo(exercise));

        if (logger.isDebugEnabled()) logger.debug("Saved Exercise");

        return po.create();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void delete(Exercise exercise) {
        if (logger.isDebugEnabled()) logger.debug("Deleting Exercise");

        UserHistoryEntityManager<ExercisePo> simpleEntityManager = new UserHistoryEntityManager<ExercisePo>(entityManager, ExercisePo.TABLE_NAME);
        simpleEntityManager.delete(exercise, ExercisePo.class);
    }
}
