package com.agilex.healthcare.mobilehealthplatform.datalayer.calendar;

import java.util.List;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.JpaFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.JpaNullFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.SimpleEntityManager;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.*;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientPublicKey;
import com.agilex.healthcare.utility.NullChecker;

@Repository
public class PatientPublicKeyDao extends AbstractDao {

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientPublicKeyDao.class);

    public static final String WHERE_PUBLIC_ID = "FROM PatientPublicKeyPo WHERE id = :id";
    public static final String WHERE_PATIENT_ID = "FROM PatientPublicKeyPo WHERE userId = :patientId";

    public PatientPublicKey getKey(String patientId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving public key for patient");

        JpaFilter jpaFilter = new JpaNullFilter();
        SimpleEntityManager<PatientPublicKeyPo> simpleEntityManager = new SimpleEntityManager<PatientPublicKeyPo>(WHERE_PATIENT_ID, entityManager);

        List<PatientPublicKeyPo> results = simpleEntityManager.get(patientId, jpaFilter, PatientPublicKeyPo.class);

        return getSinglePatientPublicKeyOrNull(results);
    }

    public PatientPublicKey getPatientId(String publicKey) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving patient id for public key");

        SimpleEntityManager<PatientPublicKeyPo> simpleEntityManager = new SimpleEntityManager<PatientPublicKeyPo>(WHERE_PUBLIC_ID, entityManager);

        List<PatientPublicKeyPo> results = simpleEntityManager.get(publicKey, PatientPublicKeyPo.class);

        return getSinglePatientPublicKeyOrNull(results);
    }

    private PatientPublicKey getSinglePatientPublicKeyOrNull(List<PatientPublicKeyPo> keys) {

        if(NullChecker.isNullish(keys)) {
            return null;
        } else {
            return keys.get(0).create();
        }
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public PatientPublicKey save(PatientPublicKey key) {
        if (logger.isDebugEnabled()) logger.debug("Saving calendar public key");

        SimpleEntityManager<PatientPublicKeyPo> simpleEntityManager = new SimpleEntityManager<PatientPublicKeyPo>(entityManager);
        PatientPublicKeyPo po = simpleEntityManager.save(new PatientPublicKeyPo(key));

        if (logger.isDebugEnabled()) logger.debug("Saved calendar public key");

        return po.create();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void delete(PatientPublicKey key) {
        if (logger.isDebugEnabled()) logger.debug("Deleting calendar public key");

        SimpleEntityManager<PatientPublicKeyPo> simpleEntityManager = new SimpleEntityManager<PatientPublicKeyPo>(entityManager);
        simpleEntityManager.delete(key, PatientPublicKeyPo.class);
    }
}
