package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresult;

import java.util.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.*;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.utility.NullSafeStringComparer;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.*;

@Repository
public class AssessmentResultDao extends AbstractDao {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(AssessmentResultDao.class);

    public static final String WHERE_DRAFT = "from AssessmentResultPo where userId = :patientId and inProgress = true";
    public static final String WHERE_COMPLETED_ASSESSMENT = "Select a from AssessmentResultPo a where a.userId = :patientId and a.assessmentId = :assessmentId and a.inProgress = false and " +
                                                            "a.dateTaken = (select max(po.dateTaken) from AssessmentResultPo po where po.assessmentId = :assessmentId and po.inProgress = false and po.userId = :patientId)";
    public static final String WHERE_ID = "from AssessmentResultPo where userId = :patientId and id = :id";
    public static final String WHERE_PATIENT_ID = "from AssessmentResultPo where userId = :patientId";
    public static final String WHERE_FINISHED_ASSESSMENTS = "from AssessmentResultPo where userId = :patientId and inProgress = false";
    
    public AssessmentResult get(PatientIdentifier patientIdentifier, String resultId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving assessment result infoß");

        SimpleEntityManager<AssessmentResultPo> simpleEntityManager = new SimpleEntityManager<AssessmentResultPo>(WHERE_ID, entityManager);
        List<AssessmentResultPo> results = simpleEntityManager.get(patientIdentifier.getUniqueId(), resultId, AssessmentResultPo.class);

        if (results.size() == 0)
            return null;

        if (results.size() > 1 && logger.isDebugEnabled())
            logger.debug("Found more than one assessment result for an id");

        return results.get(0).create();
    }

    public AssessmentResult getComplete(PatientIdentifier patientIdentifier, String assessmentId) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving assessment result info");

        AssessmentCompleteJpaFilter filter = new AssessmentCompleteJpaFilter(assessmentId);
        SimpleEntityManager<AssessmentResultPo> simpleEntityManager = new SimpleEntityManager<AssessmentResultPo>(WHERE_COMPLETED_ASSESSMENT, entityManager);
        List<AssessmentResultPo> results = simpleEntityManager.get(patientIdentifier.getUniqueId(), filter, AssessmentResultPo.class);

        if (results.size() == 0)
            return null;

        if (results.size() > 1 && logger.isDebugEnabled())
            logger.debug("Found more than one assessment result for an assessment id");

        return results.get(0).create();
    }

    public AssessmentResults getByAssessment(String patientId, String assessmentId, DateFilter filter) {
        if (logger.isDebugEnabled()) logger.debug("Retrieving assessment results for patient");

        AssessmentResults assessmentResults = get(WHERE_FINISHED_ASSESSMENTS, patientId, filter);
        
        return filter(assessmentId, assessmentResults);
    }

    public AssessmentResults getDrafts(String patientId) {
        if (logger.isDebugEnabled())
            logger.debug("Retrieving draft assessment results for a patient");

        return get(WHERE_DRAFT, patientId, DateFilterFactory.createEmptyFilter());
    }

    private AssessmentResults get(String query, String patientId, DateFilter filter) {
        JpaFilter jpaFilter = new JpaDateFilter(filter, "dateTaken");
        SimpleEntityManager<AssessmentResultPo> simpleEntityManager = new SimpleEntityManager<AssessmentResultPo>(query, entityManager);

        List<AssessmentResultPo> results = simpleEntityManager.get(patientId, jpaFilter, AssessmentResultPo.class);

        List<Transformable<? extends DomainTransferObject>> transformableList = createTransformableList(results);

        AssessmentResults assessmentResults = new AssessmentResults();
        Mapper.add(transformableList, assessmentResults);

        return assessmentResults;
    }

    private List<Transformable<? extends DomainTransferObject>> createTransformableList(List<AssessmentResultPo> results) {
        List<Transformable<? extends DomainTransferObject>> transformableList = new ArrayList<Transformable<? extends DomainTransferObject>>();
        transformableList.addAll(results);

        return transformableList;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public AssessmentResult save(AssessmentResult result) {
        if (logger.isDebugEnabled()) logger.debug("Saving assessment result");

        removePotentialExistingDraft(result);

        UserHistoryEntityManager<AssessmentResultPo> simpleEntityManager = new UserHistoryEntityManager<AssessmentResultPo>(entityManager, AssessmentResultPo.TABLE_NAME, result);
        AssessmentResultPo po = simpleEntityManager.save(new AssessmentResultPo(result));

        if (logger.isDebugEnabled())
            logger.debug("Saved assessment result");

        return po.create();
    }

    private void removePotentialExistingDraft(AssessmentResult assessmentResult) {
        entityManager.createQuery("delete AssessmentResultPo where userId = :patientId and assessmentId = :assessmentId and inProgress = true")
                .setParameter("patientId", assessmentResult.getPatientId())
                .setParameter("assessmentId", assessmentResult.getAssessmentId())
                .executeUpdate();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void delete(AssessmentResult result) {
        if (logger.isDebugEnabled()) logger.debug("Deleting Assessment Result");

        UserHistoryEntityManager<AssessmentResultPo> simpleEntityManager = new UserHistoryEntityManager<AssessmentResultPo>(entityManager, AssessmentResultPo.TABLE_NAME);
        simpleEntityManager.delete(result, AssessmentResultPo.class);
    }
    
    private AssessmentResults filter(String assessmentId, AssessmentResults results) {
    	AssessmentResults filtered = new AssessmentResults();
    	
    	for(AssessmentResult result : results) {
    		if(NullSafeStringComparer.areEqualIgnoreCase(assessmentId, result.getAssessmentId())) {
    			filtered.add(result);
    		}
    	}
    	
    	return filtered;
    }
}