package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.Query;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response.Status;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import com.agilex.healthcare.mobilehealthplatform.datalayer.AbstractDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.validPo;
import com.agilex.healthcare.mobilehealthplatform.datalayer.userhistory.UserHistoryEntityManager;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequests;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.NullChecker;

@Repository
public class AppointmentRequestDao extends AbstractDao {
	private static final org.apache.commons.logging.Log LOGGER = org.apache.commons.logging.LogFactory.getLog(AppointmentRequestDao.class);

	public AppointmentRequests getActiveAppointmentRequests(String patientId, DateFilter filter) {
		Query constructedQuery = this.constructWhereClause(patientId, filter);
		List<AppointmentRequestPo> retrievedRequests = this.executeForAppointmentRequests(constructedQuery);
		
		AppointmentRequests appointmentRequests = new AppointmentRequests();
		for (AppointmentRequestPo retrievedRequest : retrievedRequests) {
			appointmentRequests.add(retrievedRequest.create());
		}
		
		return appointmentRequests;
	}

	public AppointmentRequests getAppointmentRequests(AppointmentRequestFilter appointmentRequestFilter) {
		Query constructedQuery = this.constructWhereClause(appointmentRequestFilter);
		List<AppointmentRequestPo> retrievedRequests = this.executeForAppointmentRequests(constructedQuery);
		
		AppointmentRequests appointmentRequests = new AppointmentRequests();
		for (AppointmentRequestPo retrievedRequest : retrievedRequests) {
			appointmentRequests.add(retrievedRequest.create());
		}
		
		return appointmentRequests;
	}

	public AppointmentRequest getActiveAppointmentRequest(String patientId, String appointmentRequestId) {
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where patient.userId = :patientId and id = :appointmentRequestId and activeFlag = true ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		query.setParameter("patientId", patientId);

		AppointmentRequestPo po = this.executeForAppointmentRequest(query);
		AppointmentRequest request = (po == null)? null : po.create();
		
		return request;
	}

	public AppointmentRequest getAppointmentRequest(String patientId, String appointmentRequestId) {
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where patient.userId = :patientId and id = :appointmentRequestId ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		query.setParameter("patientId", patientId);


		AppointmentRequestPo po = this.executeForAppointmentRequest(query);
		AppointmentRequest request = (po == null)? null : po.create();
		
		return request;
	}
	
	public Set<DetailCode> getDetailCodes() {
		String selectDetailCodeQuery = "from DetailCodePo order by code";
		Query query = getQuery(selectDetailCodeQuery);
		@SuppressWarnings("unchecked")
		List<DetailCodePo> retrievedDetailCodes = query.getResultList();
		Set<DetailCode> codes = new HashSet<DetailCode>();
		
		for (DetailCodePo retrievedDetailCode : retrievedDetailCodes) {
			codes.add(retrievedDetailCode.create());
		}
		
		return codes;
	}

	public DetailCode getDetailCodeByCode(String code) {
		String selectDetailCodeQuery = "from DetailCodePo where code = :code";
		Query query = getQuery(selectDetailCodeQuery);
		query.setParameter("code", code);

		DetailCodePo detailCode = executeForDetailCode(query);
		return detailCode.create();
	}

	public AppointmentRequestMessages getAppointmentRequestMessagesByAppointmentRequestId(String patientId, String appointmentRequestId) {
		AppointmentRequestMessages messages = new AppointmentRequestMessages();
		
		AppointmentRequest parentRequest = null;
		
		try {
			parentRequest = this.getAppointmentRequest(patientId, appointmentRequestId);
		} catch (WebApplicationException e) {
			if (e.getResponse().getStatus() != Status.NOT_FOUND.getStatusCode()) {
				throw e;
			}
		}
			
		if (parentRequest != null) {
			String selectDetailCodeQuery = "from AppointmentRequestMessagePo where appointmentRequestId = :appointmentRequestId";
			Query query = getQuery(selectDetailCodeQuery);
			query.setParameter("appointmentRequestId", appointmentRequestId);
			List<AppointmentRequestMessagePo> retrievedMessages = executeForAppointmentRequestMessages(query);
			
			for (AppointmentRequestMessagePo retrievedMessage : retrievedMessages) {
				messages.add(retrievedMessage.create());
			}
		}
		
		return messages;
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public AppointmentRequestMessage saveAppointmentRequestMessage(String patientId, AppointmentRequestMessage appointmentRequestMessage, boolean isProvider) {
		// Second attempts will save two instances of the appointment request -
		// parent request and the second request.
		AppointmentRequest appointmentRequest = this.getAppointmentRequest(patientId, appointmentRequestMessage.getAppointmentRequestId());
		if(isProvider){
			appointmentRequest.setHasVeteranNewMessage(true);
		}
		else{
			appointmentRequest.setHasProviderNewMessage(true);
		}
		
        UserHistoryEntityManager<AppointmentRequestMessagePo> arMessageEntityManager = new UserHistoryEntityManager<AppointmentRequestMessagePo>(entityManager, AppointmentRequestMessagePo.TABLE_NAME, appointmentRequestMessage);
        AppointmentRequestMessagePo savedAppointmentRequestMessage = arMessageEntityManager.save(new AppointmentRequestMessagePo(appointmentRequestMessage));

        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));        
		
		return savedAppointmentRequestMessage.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public AppointmentRequest updateAppointmentRequestMessageFlag(String patientId, String appointmentRequestId, boolean isProvider){
		AppointmentRequest appointmentRequest = this.getAppointmentRequest(patientId, appointmentRequestId);
		
		if(appointmentRequest == null)
			return appointmentRequest;
		
		if(isProvider){  //If user is provider, mark as provider read all new messages
			appointmentRequest.setHasProviderNewMessage(false);
		} else {
			appointmentRequest.setHasVeteranNewMessage(false);
		}
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo savedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));        
		
		return savedAppointmentRequest.create();
	}
	
	@Transactional(propagation = Propagation.REQUIRED)
	public AppointmentRequest updateProviderSeenAppointmentRequestFlag(String patientId, String appointmentRequestId){
		AppointmentRequest appointmentRequest = this.getAppointmentRequest(patientId, appointmentRequestId);
		AppointmentRequestPo updatedAppointmentRequest = new AppointmentRequestPo();

        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
		
		if(appointmentRequest != null && appointmentRequestId.equals(appointmentRequest.getAppointmentRequestId())){
			appointmentRequest.setProviderSeenAppointmentRequest(true);
			updatedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));	
		}
		return updatedAppointmentRequest.create();
	}
	
	@Transactional(propagation = Propagation.REQUIRED)
	public AppointmentRequest updateAppointmentRequest(AppointmentRequest appointmentRequest) {
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo mergedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));    
		return mergedAppointmentRequest.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public AppointmentRequest saveAppointmentRequest(AppointmentRequest appointmentRequest) {
		// Second attempts will save two instances of the appointment request -
		// parent request and the second request.
		AppointmentRequest existingAppointmentRequest = this.getAppointmentRequestIfItExists(appointmentRequest.getAppointmentRequestId());
		if (existingAppointmentRequest != null) {
			appointmentRequest.setHasProviderNewMessage(existingAppointmentRequest.isHasProviderNewMessage());
			appointmentRequest.setHasVeteranNewMessage(existingAppointmentRequest.isHasVeteranNewMessage());
		}
        UserHistoryEntityManager<AppointmentRequestPo> simpleEntityManager = new UserHistoryEntityManager<AppointmentRequestPo>(entityManager, AppointmentRequestPo.TABLE_NAME, appointmentRequest);
        AppointmentRequestPo savedAppointmentRequest = simpleEntityManager.save(new AppointmentRequestPo(appointmentRequest));
        if (savedAppointmentRequest != null) {
    		return savedAppointmentRequest.create();
        }
        return null;
	}
	
	public AppointmentRequest getAppointmentRequestIfItExists(String appointmentRequestId) {
		AppointmentRequestPo arpo = null;
		String selectSingleAppointmentRequest = "from AppointmentRequestPo where id = :appointmentRequestId ";
		Query query = getQuery(selectSingleAppointmentRequest);
		query.setParameter("appointmentRequestId", appointmentRequestId);
		try{
			arpo = (AppointmentRequestPo) query.getSingleResult();
		} catch(NoResultException e) {
			LOGGER.debug(e);		
			arpo = null;
		} catch (NonUniqueResultException e) {
			LOGGER.debug(e);	
			arpo = null;
		}
		return (arpo == null) ? null : arpo.create();
	}

	@Transactional(propagation = Propagation.REQUIRED)
	public void deleteAppointmentRequest(AppointmentRequest appointmentRequest) {
		validPo<AppointmentRequestPo> po = new validPo<AppointmentRequestPo>(entityManager, new AppointmentRequestPo(appointmentRequest));
		if (po.isOk()) {
			AppointmentRequestPo appointmentRequestToRemove = po.getPo();
			appointmentRequestToRemove.setDeletedDate(new Date());
			appointmentRequestToRemove.setActiveFlag(false);
			this.updateAppointmentRequest(appointmentRequestToRemove.create());
		}
	}

	private Query constructWhereClause(String patientId, DateFilter filter) {
		StringBuilder whereClause = new StringBuilder();

		Map<String, Object> parameters = new HashMap<String, Object>();

		whereClause.append(" where 1=1 ");

		addPatientIdCriteria(whereClause);
		addActiveRecordCriteria(whereClause);
		addDateFilterCriteria(filter, whereClause);

		whereClause.append(" order by hasVeteranNewMessage DESC, lastUpdatedDate DESC");

		setPatientIdParameter(patientId, parameters);
		setActiveFlagParameter(parameters);
		setDataFilterParameter(filter, parameters);

		String query = "from AppointmentRequestPo " + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private void setPatientIdParameter(String patientId, Map<String, Object> parameters) {
		parameters.put("patientId", patientId);
	}

	private void addPatientIdCriteria(StringBuilder whereClause) {
		whereClause.append(" and patient.userId = :patientId ");
	}

	private void setDataFilterParameter(DateFilter filter, Map<String, Object> parameters) {

		if (filter == null) {
			return;
		}

		if (NullChecker.isNotNullish(filter.getStartDate())) {
			parameters.put("startDate", filter.getStartDate());
		}
		if (NullChecker.isNotNullish(filter.getEndDate())) {
			parameters.put("endDate", filter.getEndDate());
		}
	}

	private void addDateFilterCriteria(DateFilter filter, StringBuilder whereClause) {

		if (filter == null) {
			return;
		}

		if (NullChecker.isNotNullish(filter.getStartDate())) {
			whereClause.append(" and createdDate >= :startDate ");
		}
		if (NullChecker.isNotNullish(filter.getEndDate())) {
			whereClause.append(" and createdDate <= :endDate ");
		}
	}

	private Query constructWhereClause(AppointmentRequestFilter filter) {

		StringBuilder whereClause = new StringBuilder();
		Map<String, Object> parameters = new HashMap<String, Object>();

		whereClause.append(" where 1=1 ");

		addActiveRecordCriteria(whereClause);
		addFacilityNameCriteria(filter, whereClause);
		addParentSiteCodeCriteria(filter, whereClause);
		addDateFilterCriteria(filter, whereClause);
		
		whereClause.append(" order by case when hasProviderNewMessage=1 and providerSeenAppointmentRequest=1 then  hasProviderNewMessage  end DESC, lastUpdatedDate ASC");

		setActiveFlagParameter(parameters);
		setFacilityNameParameter(filter, parameters);
		setParentSiteCodeParameter(filter, parameters);
		setDataFilterParameter(filter, parameters);

		String query = "from AppointmentRequestPo " + whereClause.toString();
		Query constructedQuery = this.setParametersForQuery(query, parameters);
		return constructedQuery;
	}

	private void setFacilityNameParameter(AppointmentRequestFilter filter, Map<String, Object> parameters) {
		if (filter != null && NullChecker.isNotNullish(filter.getFacilityName())) {
			parameters.put("facilityName", filter.getFacilityName().toLowerCase());
		}
	}

	private void setParentSiteCodeParameter(AppointmentRequestFilter filter, Map<String, Object> parameters) {
		if (filter != null && NullChecker.isNotNullish(filter.getParentSiteCode())) {
			parameters.put("parentSiteCode", filter.getParentSiteCode());
		}
	}

	private void setActiveFlagParameter(Map<String, Object> parameters) {
		parameters.put("activeFlag", true);
	}

	private void addParentSiteCodeCriteria(AppointmentRequestFilter filter, StringBuilder whereClause) {
		if (filter != null && NullChecker.isNotNullish(filter.getParentSiteCode())) {
			whereClause.append(" and facility.parentSiteCode = :parentSiteCode ");
		}
	}

	private void addFacilityNameCriteria(AppointmentRequestFilter filter, StringBuilder whereClause) {
		if (filter != null && NullChecker.isNotNullish(filter.getFacilityName())) {
			whereClause.append(" and lower(facility.name) = :facilityName ");
		}
	}

	private void addActiveRecordCriteria(StringBuilder whereClause) {
		whereClause.append(" and activeFlag = :activeFlag ");
	}

	private Query setParametersForQuery(String query, Map<String, Object> parameters) {
		//LOGGER.debug(String.format(":::: Query: %s", query));
		Query constructedQuery = this.entityManager.createQuery(query);
		for (String key : parameters.keySet()) {
			constructedQuery.setParameter(key, parameters.get(key));
		}
		return constructedQuery;
	}

	private List<AppointmentRequestPo> executeForAppointmentRequests(Query query) {
		@SuppressWarnings("unchecked")
		List<AppointmentRequestPo> appointmentRequestResults = query.getResultList();

		return appointmentRequestResults;
	}

	private List<AppointmentRequestMessagePo> executeForAppointmentRequestMessages(Query query) {
		@SuppressWarnings("unchecked")
		List<AppointmentRequestMessagePo> appointmentRequestMessageResults = query.getResultList();

		return appointmentRequestMessageResults;
	}

	private AppointmentRequestPo executeForAppointmentRequest(Query query) {
		try {
			return (AppointmentRequestPo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private DetailCodePo executeForDetailCode(Query query) {
		try {
			return (DetailCodePo) query.getSingleResult();
		} catch (NoResultException e) {
			throw new WebApplicationException(Status.NOT_FOUND);
		} catch (NonUniqueResultException e) {
			throw new WebApplicationException(Status.PRECONDITION_FAILED);
		}
	}

	private Query getQuery(String dbQuery) {
		Query query = this.entityManager.createQuery(dbQuery);
		return query;
	}

}
