package com.agilex.healthcare.mobilehealthplatform.repository.dao.rightofaccess;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;

import javax.annotation.Resource;

import com.agilex.healthcare.mobilehealthplatform.datalayer.rightofaccess.RightOfAccessDao;
import org.apache.commons.io.IOUtils;
import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.SpringBasedIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.datalayer.rightofaccess.RightOfAccessPo;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class RightOfAccessDaoTest extends SpringBasedIntegrationRepo {
	private String acceptedUserId = "testAccepted";
	private String notAcceptedUserId = "testNotAccepted";
	private String RESOURCE_DIR = "com/agilex/healthcare/mobilehealthplatform/repository/dao/rightofaccess/";
	byte[] roaImage = null;
	
	@Resource
    RightOfAccessDao dao;
	
	@BeforeClass
	public static void checkWhetherTestsAreRunningInVaVeteranDevMode() {
		Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
	}
	
	@Before
	public void setUp() {
		dao.saveRightOfAccess(generateRightOfAccessInfo(true));
		dao.saveRightOfAccess(generateRightOfAccessInfo(false));		
	}
	
	@Test
	public void hasAcceptedRightOfAccess() {
		boolean isAccepted = dao.hasAcceptedRightOfAccess(acceptedUserId);
		assertEquals(true, isAccepted);
	}

	@Test
	public void hasNotAcceptedRightOfAccess() {
		boolean isAccepted = dao.hasAcceptedRightOfAccess(notAcceptedUserId);
		assertEquals(false, isAccepted);
	}

	@Test
	public void getRightOfAccessAccepted() {
		RightOfAccessPo info = dao.getRightOfAccess(acceptedUserId);
		assertEquals(true, info.isRightOfAccessAccepted());
		assertNotNull(info.getRightOfAccessDate());
		assertNotNull(info.getRightOfAccessPDF());
	}

	@Test
	public void getRightOfAccessNotAccepted() {
		RightOfAccessPo info = dao.getRightOfAccess(notAcceptedUserId);
		assertEquals(false, info.isRightOfAccessAccepted());
		assertNull(info.getRightOfAccessPDF());
	}

	@Test
	public void testSaveRightOfAccessInfo() {
		RightOfAccessPo rightOfAccess = generateRightOfAccessInfo(true);
		RightOfAccessPo savedRightOfAccess = dao.saveRightOfAccess(rightOfAccess);
		compareRightOfAccess(rightOfAccess, savedRightOfAccess);

		RightOfAccessPo retrievedRightOfAccess = dao.getRightOfAccess(acceptedUserId);
		compareRightOfAccess(rightOfAccess, retrievedRightOfAccess);
	}
	
	@Test
	public void testSaveRightOfAccessInfoNotAccepted() {
		RightOfAccessPo rightOfAccess = generateRightOfAccessInfo(false);
		RightOfAccessPo savedRightOfAccessInfo = dao.saveRightOfAccess(rightOfAccess);
		compareRightOfAccess(rightOfAccess, savedRightOfAccessInfo);

		RightOfAccessPo retrievedRightOfAccess = dao.getRightOfAccess(notAcceptedUserId);
		compareRightOfAccess(rightOfAccess, retrievedRightOfAccess);
	}

	private RightOfAccessPo generateRightOfAccessInfo(boolean accepted) {
		String userId = notAcceptedUserId;
		byte[] roaImage = null;

		if(accepted) {
			userId = acceptedUserId;
			roaImage = readROAPDFFile();
		}
		
		RightOfAccessPo info = new RightOfAccessPo(userId, accepted, new Date(), roaImage);
		return info;
	}
	
	private byte [] readROAPDFFile(){
		InputStream roaImageStream = this.getClass().getClassLoader().getResourceAsStream(RESOURCE_DIR + "ROA.pdf");
		
		if (roaImage == null){
			try {
				roaImage = IOUtils.toByteArray(roaImageStream);
			} catch (IOException e) {
				throw new RuntimeException(e);
			}
		}
		return roaImage;
	}
	
	private void compareRightOfAccess(RightOfAccessPo expected, RightOfAccessPo actual) {
		assertEquals(expected.getUserId(), actual.getUserId());
		assertEquals(expected.isRightOfAccessAccepted(), actual.isRightOfAccessAccepted());
		Date expectedDay = expected.getRightOfAccessDate();
		Date actualDay = actual.getRightOfAccessDate();
		assertTrue(DateHelper.calculateDeltaInDays(expectedDay, actualDay) == 0);
	}	
}
