package com.agilex.healthcare.mobilehealthplatform.repository.dao.audit;

import java.util.Date;
import java.util.List;
import java.util.UUID;

import javax.annotation.Resource;

import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.SpringBasedIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogDao;
import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogMapper;
import com.agilex.healthcare.mobilehealthplatform.datalayer.audit.AuditLogPo;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLog;
import com.agilex.healthcare.mobilehealthplatform.domain.AuditLogs;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.ModeHelper;

public class AuditLogDaoTest extends SpringBasedIntegrationRepo {
	@Resource
    AuditLogDao dao;
	AuditLogMapper mapper = new AuditLogMapper();

    @BeforeClass
    public static void checkWhetherTestsAreRunningInVeteranDevMode() {
        Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
    }

	@Test
	public void verifySaveAuditLog() {
		String testUserId = UUID.randomUUID().toString();
		AuditLogPo savedAuditLog = saveAuditLog(testUserId);

		assertNotNull(savedAuditLog);
		assertNotNull(savedAuditLog.getId());

		List<AuditLogPo> retrievedAuditLogs = dao.fetchAuditLogs(testUserId);
		
		assertNotNull(retrievedAuditLogs);
		assertEquals(1, retrievedAuditLogs.size());
		compareAuditLogs(savedAuditLog, retrievedAuditLogs.get(0));
	}

	@Test
	public void verifyFetchAuditLogs() {
		String testUserId = UUID.randomUUID().toString();
		saveAuditLogs(testUserId, 10);

		List<AuditLogPo> retrievedAuditLogs = dao.fetchAuditLogs(testUserId);

		assertNotNull(retrievedAuditLogs);
		assertEquals(10, retrievedAuditLogs.size());
	}

	@Test
	public void verifyFetchAuditLog() {
		String testUserId = UUID.randomUUID().toString();
		AuditLogPo savedAuditLog = saveAuditLog(testUserId);
		
		AuditLogPo fetchedLog = dao.fetchAuditLog(testUserId, savedAuditLog.getId());
		assertNotNull(fetchedLog);
		assertNotNull(savedAuditLog);
		assertNotNull(savedAuditLog.getId());
		compareAuditLogs(savedAuditLog, fetchedLog);
	}
	
	@Test
	public void verifyFetchAuditLogWithBadIdReturnsNull() {
		String userId = UUID.randomUUID().toString();
		AuditLogPo savedAuditLog = saveAuditLog(userId);
		assertNotNull(savedAuditLog);
		
		AuditLogPo fetchedLog = dao.fetchAuditLog(userId, UUID.randomUUID().toString());
		assertNull(fetchedLog);
	}

	@Test
	public void verifyGetAuditLogsIsEmptyForNewUser() {
		String testUserId = UUID.randomUUID().toString();
		saveAuditLogs(testUserId, 10);

		List<AuditLogPo> retrievedAuditLogs = dao.fetchAuditLogs("non-existant-user");

		assertEquals(0, retrievedAuditLogs.size());
	}

	@Test
	public void verifyGetAuditLogsByUserId() {
		String testUserId = UUID.randomUUID().toString();
		String testUserId2 = UUID.randomUUID().toString();
		saveAuditLogs(testUserId, 10);
		saveAuditLogs(testUserId2, 10);

		List<AuditLogPo> retrievedAuditLogs = dao.fetchAuditLogs(testUserId2);
		AuditLogs auditLogs = new AuditLogs();
		mapper.toDtoCollection(retrievedAuditLogs, auditLogs);

		assertNotNull(auditLogs);
		assertTrue(auditLogs.size() == 10);
	}

	private void saveAuditLogs(String testUserId, int numberToGenerate) {
		for (int i = 0; i < numberToGenerate; i++) {
			AuditLog auditLogToSave = new AuditLog(testUserId, "", "", new Date(), "");
			AuditLogPo po = mapper.toPo(auditLogToSave);
			dao.saveAuditLog(po);
		}
	}

	private AuditLogPo saveAuditLog(String testUserId) {
		AuditLog auditLogToSave = new AuditLog(testUserId, "", "", new Date(), "");
		AuditLogPo po = mapper.toPo(auditLogToSave);
		return dao.saveAuditLog(po);
	}

	private void compareAuditLogs(AuditLogPo expected, AuditLogPo actual) {
		compareStringAllowNull(expected.getId(), actual.getId());
		compareStringAllowNull(expected.getUserId(), actual.getUserId());
		compareStringAllowNull(expected.getType(), actual.getType());
		compareStringAllowNull(expected.getSubType(), actual.getSubType());
		assertEquals(0, DateHelper.calculateDeltaInDays(expected.getDate(), actual.getDate()));
		compareStringAllowNull(expected.getDetails(), actual.getDetails());
	}
	
	private void compareStringAllowNull(String expected, String actual) {
		assertEquals(convertNullToEmptyString(expected), convertNullToEmptyString(actual));
	}
	
	private String convertNullToEmptyString(String value) {
		if (value == null) {
			return "";
		}
		return value;
	}
}