package com.agilex.healthcare.mobilehealthplatform.datalayer.vital;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.junit.After;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.datalayer.vitals.VitalDataLayerRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.utility.DateHelper;

public class VitalDataLayerRepoTest extends UserHistoryIntegrationRepo {

    @Resource
    VitalDataLayerRepo vitalDataLayer;

    private static final String integrationTestPatientId = "integrationTestPatientId";

    @After
    public void cleanup() {
        VitalEntries entries = vitalDataLayer.getVitalEntries(integrationTestPatientId);
        for(VitalEntry entry : entries) {
            vitalDataLayer.deleteVitalEntry(entry);
        }
    }

    @Test
	public void testGetAllVitalEntries() {
        createSaveEntry();
        createSaveEntry();

		VitalEntries vitalEntries = vitalDataLayer.getVitalEntries(integrationTestPatientId);
		assertNotNull(vitalEntries);
        assertEquals(2, vitalEntries.size());
        assertEquals(2, vitalEntries.get(0).getVitalObservations().size());
        assertEquals(2, vitalEntries.get(1).getVitalObservations().size());
    }

    @Test
	public void testSingleVitalEntryRetrieval() {
        createSaveEntry();

		VitalEntries vitalEntries = vitalDataLayer.getVitalEntries(integrationTestPatientId);
		assertNotNull(vitalEntries);
        assertEquals(1, vitalEntries.size());
        VitalEntry vitalEntry = vitalEntries.get(0);
        VitalEntry actualVitalEntry = vitalDataLayer.getVitalEntry(integrationTestPatientId, vitalEntry.getId());
        assertEquals(vitalEntry.getId(), actualVitalEntry.getId());
        
    }
    
    @Test
	public void testGetVitalEntryByStartDateAndEndDate() {
		VitalEntry entry = new VitalEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setQualifier("Standing");
		entry.setSection("BP");

		VitalObservation observation = new VitalObservation();
		observation.setType("8867-4");
		observation.setValue("60");
		observation.setValueUnits("/min");
		observation.setVitalEntry(entry);
		entry.getVitalObservations().add(observation);

		observation = new VitalObservation();
		observation.setType("8480-6");
		observation.setValue("80");
		observation.setValueUnits("mm(hg)");
		observation.setVitalEntry(entry);
		entry.getVitalObservations().add(observation);

		try {
			Thread.sleep(1000);
		} catch (InterruptedException occassionalPersistenceDelayCausesFailures) {
		}

		VitalEntry vitalEntry = vitalDataLayer.saveVitalEntry(entry);
		assertNotNull(vitalEntry.getId());
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setStartDate(getYesterday());
		filter.setEndDate(new Date());
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter.setStartDate(new Date());
		VitalEntries shouldBeEmpty = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
	}

	@Test
	public void testGetVitalEntryByStartDateEndDateAndSection() {
        createSaveEntry();
		
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setStartDate(getYesterday());
		filter.setEndDate(getTomorrow());
		filter.setSection("BP");
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter.setSection("BAD");
		VitalEntries shouldBeEmpty = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	@Test
	public void testGetVitalEntryBySection() {
        createSaveEntry();
		
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setSection("BP");
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter.setSection("BAD");
		VitalEntries shouldBeEmpty = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	@Test
	public void testGetVitalEntryByStartDateAndSection() {
		this.createAndSaveVitalEntry(integrationTestPatientId, "BP", DateHelper.parseDate("1/1/2012"));
		this.createAndSaveVitalEntry(integrationTestPatientId, "BP", DateHelper.parseDate("1/2/2012"));
				
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setSection("BP");
		filter.setStartDate(DateHelper.parseDate("1/2/2012"));
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter.setSection("BAD");
		VitalEntries shouldBeEmpty = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
		
	}
	
	@Test
	public void verifyUpdate() {
		VitalEntry entry = createSaveEntry();

		List<VitalObservation> vitalObservations = entry.getVitalObservations();
		assertNotNull(vitalObservations);
		assertEquals(2, vitalObservations.size());
		
		VitalObservation observation = vitalObservations.get(0);
		observation.setType("MOD");
		
		entry = vitalDataLayer.saveVitalEntry(entry);
		
		vitalObservations = entry.getVitalObservations();
		assertNotNull(vitalObservations);
		assertEquals(2, vitalObservations.size());
		
		observation = vitalObservations.get(0);
		assertEquals("MOD", observation.getType());
	}
	
	@Test
	public void testGetVitalEntryByEndDateAndSection() {
        createSaveEntry();
		
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setSection("BP");
		filter.setEndDate(getTomorrow());
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
	
		filter.setSection("BP");
		filter.setEndDate(getYesterday());
		VitalEntries shouldBeZero = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeZero.size());
		
		filter.setSection("BAD");
		filter.setEndDate(getTomorrow());
		VitalEntries shouldBeEmpty = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
		
		filter.setSection("BAD");
		filter.setEndDate(getYesterday());
		VitalEntries shouldBeZilch = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeZilch.size());
	}
	
	@Test
	public void testGetVitalEntryByEndDate() {
        createSaveEntry();
		
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setEndDate(getTomorrow());
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter.setEndDate(getYesterday());
		VitalEntries shouldBeZilch = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeZilch.size());
	}

	@Test
	public void testGetVitalEntryByStartDate() {
		this.createAndSaveVitalEntry(integrationTestPatientId, "BP", DateHelper.parseDate("1/1/2012"));
		this.createAndSaveVitalEntry(integrationTestPatientId, "BP", DateHelper.parseDate("1/2/2012"));
		this.createAndSaveVitalEntry(integrationTestPatientId, "BP", DateHelper.parseDate("1/3/2012"));
		
		VitalEntryFilter filter = new VitalEntryFilter();
		filter.setSection("BP");
		filter.setStartDate(DateHelper.parseDate("1/1/2012"));
		VitalEntries shouldBeThree = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(3, shouldBeThree.size());
		
		filter.setStartDate(DateHelper.parseDate("1/3/2012"));
		VitalEntries shouldBeOne = vitalDataLayer.getVitalEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
	}

    private VitalEntry createSaveEntry() {
        VitalEntry entry = new VitalEntry();
        entry.setEntryDate(new Date());
        entry.setNotes("itest notes");
        entry.setPatientId(integrationTestPatientId);
        entry.setQualifier("Standing");
        entry.setSection("BP");

        VitalObservation observation = new VitalObservation();
        observation.setType("8867-4");
        observation.setValue("60");
        observation.setValueUnits("/min");
        observation.setVitalEntry(entry);
        entry.getVitalObservations().add(observation);

        observation = new VitalObservation();
        observation.setType("8480-6");
        observation.setValue("80");
        observation.setValueUnits("mm(hg)");
        observation.setVitalEntry(entry);
        entry.getVitalObservations().add(observation);

        return vitalDataLayer.saveVitalEntry(entry);
    }

    private VitalEntry createAndSaveVitalEntry(String patientId, String section, Date date) {
		VitalEntry entryToSave = createVitalEntry(patientId, section, date);
		return vitalDataLayer.saveVitalEntry(entryToSave);
	}

    private VitalEntry createVitalEntry(String patientId, String section, Date date) {
        VitalEntry entry = new VitalEntry();
        entry.setEntryDate(date);
        entry.setNotes("itest notes");
        entry.setPatientId(patientId);
        entry.setQualifier("Standing");
        entry.setSection(section);

        VitalObservation observation = new VitalObservation();
        observation.setType("8867-4"); // pulse
        observation.setValue("60");
        observation.setValueUnits("/min");
        observation.setVitalEntry(entry);
        entry.getVitalObservations().add(observation);

        return entry;
    }

    private Date getYesterday() {
		Calendar cal = Calendar.getInstance();
		DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

		cal.add(Calendar.DATE, -1);
		return cal.getTime();
	}
	
	private Date getTomorrow() {
		Calendar cal = Calendar.getInstance();
		DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

		cal.add(Calendar.DATE, +1);
		return cal.getTime();
	}
}
