package com.agilex.healthcare.mobilehealthplatform.datalayer.mood;

import java.util.*;

import javax.annotation.Resource;

import org.joda.time.DateTime;
import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;

public class MoodDataLayerRepoTest extends UserHistoryIntegrationRepo {

    public static final String NOTES = "test note.";
    public static final String VALUE = "no worries";

    @Resource
    MoodEventDataLayerRepo dataLayer;

    MoodEvent mood;
    String patientId;
    Date entryDate;


    @Before
    public void before() {
        patientId = UUID.randomUUID().toString();
        entryDate = new Date();

        createMood();
    }

    private void createMood() {
        mood = new MoodEvent();
        mood.setPatientId(patientId);
        mood.setDate(entryDate);
        mood.setNotes(NOTES);
        mood.setValue(VALUE);
    }

    @After
    public void after() {
        dataLayer.deleteMoodEvent(mood);
    }

    @Test
    public void updateMood() {
        saveFilledInMood();
        saveEmptyMood();
    }

 
    @Test
    public void findByPatientId() {
        saveFilledInMood();

        MoodEvents moodEvents = dataLayer.getMoodEvents(patientId);
        assertEquals(1, moodEvents.getSize());
        verifyMoodOptionalData(moodEvents.get(0));
    }
    
    @Test
    public void validateRetrieveMoodOnlyByOwner() {
    	saveFilledInMood();
    	
    	MoodEvent retrieved = dataLayer.getMoodEvent(patientId, mood.getUniqueId());
    	assertNotNull(retrieved);
    	
    	String otherUser = UUID.randomUUID().toString();
    	dataLayer.getMoodEvent(otherUser, mood.getUniqueId());
    	
    }

    @Test
    public void findBeforeToday() {
        saveFilledInMood();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(entryDate), null);
        MoodEvents moodEvents = dataLayer.getMoodEvents(patientId, filter);

        assertEquals(1, moodEvents.getSize());
        verifyMoodOptionalData(moodEvents.get(0));
    }

    @Test
    public void findAfterToday() {
        saveFilledInMood();

        DateFilter filter = DateFilterFactory.createFilterFromDate(null, createPlusOne(entryDate));
        MoodEvents moodEvents = dataLayer.getMoodEvents(patientId, filter);

        assertEquals(1, moodEvents.getSize());
        verifyMoodOptionalData(moodEvents.get(0));
    }

    @Test
    public void findBetweenDates() {
        saveFilledInMood();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(entryDate), createPlusOne(entryDate));
        MoodEvents moodEvents = dataLayer.getMoodEvents(patientId, filter);

        assertEquals(1, moodEvents.getSize());
        verifyMoodOptionalData(moodEvents.get(0));
    }

    private void saveEmptyMood() {
        mood.setDate(null);
        mood.setNotes(null);
        mood.setValue(null);
        saveFilledInMood();

        assertNull(mood.getDate());
        assertNull(mood.getNotes());
        assertNull(mood.getValue());
    }

    private void saveFilledInMood() {
        mood = dataLayer.saveMoodEvent(mood);
        assertNotNull(mood);
        assertEquals(patientId, mood.getPatientId());
    }

    private void verifyMoodOptionalData(MoodEvent singleMood) {
        assertEquals(NOTES, singleMood.getNotes());
        assertEquals(VALUE, singleMood.getValue());
    }

    private Date createPlusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.plusDays(1).toDate();
    }

    private Date createMinusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.minusDays(1).toDate();
    }

    @Test
    public void noMoodFoundForPatient() {
        saveFilledInMood();

        MoodEvents moodEvents = dataLayer.getMoodEvents("1");
        assertEquals(0, moodEvents.getSize());
    }

    @Test
    public void noMoodFoundForId() {
        saveFilledInMood();

        MoodEvent moodEvent = dataLayer.getMoodEvent(null, "1");
        assertNull(moodEvent);
    }

    @Test
    public void noMoodFoundBetweenDates() {
        saveFilledInMood();

        DateTime now = new DateTime(entryDate);
        Date tenDaysAgo = now.minusDays(10).toDate();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(tenDaysAgo), createPlusOne(tenDaysAgo));

        MoodEvents moodEvents = dataLayer.getMoodEvents(patientId, filter);
        assertEquals(0, moodEvents.getSize());
    }
}
