package com.agilex.healthcare.mobilehealthplatform.datalayer.medication.rxrefill;

import java.util.*;

import javax.annotation.Resource;

import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.SpringBasedIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationRefillRequestDataLayer;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.utility.DateHelper;
import com.google.common.collect.Lists;

public class MedicationRefillRequestDataLayerTest extends SpringBasedIntegrationRepo {

	@Resource
	MedicationRefillRequestDataLayer refillRequestRepo;
	
	MedicationRefillRequest refillRequest;
	String medicationId;
	String patientId;
	Date requestDate;
	
	@Before
	public void before() {
		medicationId = UUID.randomUUID().toString();
		patientId = UUID.randomUUID().toString();
		requestDate = new Date();
		refillRequest = createRefillRequest(medicationId, new Date(), patientId);
	}
	
	@Test
	public void validateSave() {
		Assert.assertNotNull(refillRequest.getUniqueId());
		Assert.assertEquals(patientId, refillRequest.getPatientIdentifier().getUniqueId());
		Assert.assertEquals(requestDate, refillRequest.getRequestDate());
		Assert.assertEquals(medicationId, refillRequest.getMedicationId());
	}
	
	@Test
	public void validateGetByIdAndPatient() {
		MedicationRefillRequest retrieved = refillRequestRepo.getRefillRequest(new PatientIdentifier("", patientId), refillRequest.getMedicationId());
		assertEqualsRefillRequest(refillRequest, retrieved);
	}
	
	@Test
	public void validateGetByIdAndPatientAfterMultipleRequests() {
		createRefillRequest(medicationId, DateHelper.getToday(), patientId);
		createRefillRequest(medicationId, DateHelper.xDaysFromNow(10), patientId);
		MedicationRefillRequest refillRequestThatShouldBeReturned = createRefillRequest(medicationId, DateHelper.xDaysFromNow(20), patientId);
		MedicationRefillRequest retrieved = refillRequestRepo.getRefillRequest(new PatientIdentifier("", patientId), medicationId);
		assertEqualsRefillRequest(refillRequestThatShouldBeReturned, retrieved);
	}
	
	@Test
	public void validateFetchRecent() {
		createRefillRequest(medicationId, DateHelper.xDaysFromNow(10), patientId);
		MedicationRefillRequest expectedRefillRequest = createRefillRequest(medicationId, DateHelper.xDaysFromNow(20), patientId);
		
		ArrayList<String> medicationIds = Lists.newArrayList(medicationId, "invalid");
		MedicationRefillRequests latestRefillRequests = refillRequestRepo.fetchLatestRefillRequests(new PatientIdentifier("", patientId), medicationIds);
		assertEquals(1, latestRefillRequests.size());
		
		MedicationRefillRequest actual = latestRefillRequests.get(0);
		assertEqualsRefillRequest(expectedRefillRequest, actual);
	}
	
	@Test
	public void validateFetchRecentWithEmptyMedicationList() {
		createRefillRequest(medicationId, DateHelper.xDaysFromNow(10), patientId);
		MedicationRefillRequest expectedRefillRequest = createRefillRequest(medicationId, DateHelper.xDaysFromNow(20), patientId);
		
		ArrayList<String> medicationIds = new ArrayList<String>();
		MedicationRefillRequests latestRefillRequests = refillRequestRepo.fetchLatestRefillRequests(new PatientIdentifier("", patientId), medicationIds);
		assertEquals(0, latestRefillRequests.size());
	}
	
	@Test
	public void validateGetByPatient() {
		MedicationRefillRequests refillRequests = refillRequestRepo.getRefillRequests(new PatientIdentifier("", patientId), DateFilterFactory.createEmptyFilter());
		Assert.assertTrue(refillRequests != null && refillRequests.size() > 0);
	}
	
	@Test
	public void validateGetByPatientWithDate() {
		MedicationRefillRequest refillRequest = new MedicationRefillRequest();
		refillRequest.setPatientId("get-by-date");
		refillRequest.setRequestDate(DateHelper.get120DaysFromNow());
		refillRequest.setMedicationId("medication-id");
		refillRequestRepo.saveRefillRequest(refillRequest);
		
		MedicationRefillRequests refillRequests = refillRequestRepo.getRefillRequests(new PatientIdentifier("", "get-by-date"), DateFilterFactory.createFilterFromDate(new Date(), DateHelper.get120DaysFromNow()));
		Assert.assertEquals(1, refillRequests.size());
		
		refillRequests = refillRequestRepo.getRefillRequests(new PatientIdentifier("", "get-by-date"), DateFilterFactory.createFilterFromDate(new Date(), DateHelper.get30DaysFromNow()));
		Assert.assertEquals(0, refillRequests.size());
	}
	
	
	MedicationRefillRequest createRefillRequest(String medicationId, Date requestDate, String patientId) {
		MedicationRefillRequest refillRequest = new MedicationRefillRequest();
		
		refillRequest.setMedicationId(medicationId);
		refillRequest.setRequestDate(requestDate);
		refillRequest.setPatientId(patientId);
		
		return refillRequestRepo.saveRefillRequest(refillRequest);
	}
	
	
	void assertEqualsRefillRequest(MedicationRefillRequest expected, MedicationRefillRequest actual) {
		Assert.assertEquals(expected.getMedicationId(), actual.getMedicationId());
		Assert.assertEquals(expected.getPatientId(), actual.getPatientId());
		Assert.assertEquals(expected.getId(), actual.getId());
		Assert.assertEquals(expected.getRequestDate(), actual.getRequestDate());
	}
}
