package com.agilex.healthcare.mobilehealthplatform.datalayer.event;

import java.util.*;

import javax.annotation.Resource;

import org.joda.time.DateTime;
import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;

public class DailyEventDataLayerRepoTest extends UserHistoryIntegrationRepo {
    public static final String NOTES = "Forty two";
    public static final String TITLE = "The meaning of life";

    @Resource
    DailyEventDataLayerRepo dataLayer;

    DailyEvent dailyEvent;
    String patientId;
    Date entryDate;


    @Before
    public void before() {
        patientId = UUID.randomUUID().toString();
        entryDate = new Date();

        createDailyEvent();
    }

    private void createDailyEvent() {
        dailyEvent = new DailyEvent();
        dailyEvent.setPatientId(patientId);
        dailyEvent.setEntryDate(entryDate);
        dailyEvent.setNotes(NOTES);
        dailyEvent.setTitle(TITLE);
    }

    @After
    public void after() {
        dataLayer.deleteDailyEvent(dailyEvent);
        DailyEvent nullDailyEvent = dataLayer.getDailyEvent(null, dailyEvent.getUniqueId());

        assertNull(nullDailyEvent);
    }

    @Test
    public void updateDailyEvent() {
        saveFilledInDailyEvent();
        saveEmptyDailyEvent();
    }

    @Test
    public void verifyOnlySelf() {
        saveFilledInDailyEvent();

        DailyEvent singleDailyEvent = dataLayer.getDailyEvent(patientId, dailyEvent.getUniqueId());
        verifyDailyEventOptionalData(singleDailyEvent);
        
        singleDailyEvent = dataLayer.getDailyEvent(patientId, dailyEvent.getUniqueId());
        assertNotNull(singleDailyEvent);
    }

    @Test
    public void findByPatientId() {
        saveFilledInDailyEvent();

        DailyEvents dailyEvents = dataLayer.getDailyEvents(patientId);
        assertEquals(1, dailyEvents.getSize());
        verifyDailyEventOptionalData(dailyEvents.get(0));
    }

    @Test
    public void findBeforeToday() {
        saveFilledInDailyEvent();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(entryDate), null);
        DailyEvents dailyEvents = dataLayer.getDailyEvents(patientId, filter);

        assertEquals(1, dailyEvents.getSize());
        verifyDailyEventOptionalData(dailyEvents.get(0));
    }

    @Test
    public void findAfterToday() {
        saveFilledInDailyEvent();

        DateFilter filter = DateFilterFactory.createFilterFromDate(null, createPlusOne(entryDate));
        DailyEvents dailyEvents = dataLayer.getDailyEvents(patientId, filter);

        assertEquals(1, dailyEvents.getSize());
        verifyDailyEventOptionalData(dailyEvents.get(0));
    }

    @Test
    public void findBetweenDates() {
        saveFilledInDailyEvent();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(entryDate), createPlusOne(entryDate));
        DailyEvents dailyEvents = dataLayer.getDailyEvents(patientId, filter);

        assertEquals(1, dailyEvents.getSize());
        verifyDailyEventOptionalData(dailyEvents.get(0));
    }

    private void saveEmptyDailyEvent() {
        dailyEvent.setEntryDate(null);
        dailyEvent.setNotes(null);
        dailyEvent.setTitle(null);
        saveFilledInDailyEvent();

        assertNull(dailyEvent.getDate());
        assertNull(dailyEvent.getNotes());
        assertNull(dailyEvent.getTitle());
    }

    private void saveFilledInDailyEvent() {
        dailyEvent = dataLayer.saveDailyEvent(dailyEvent);
        assertNotNull(dailyEvent);
        assertEquals(patientId, dailyEvent.getPatientId());
    }

    private void verifyDailyEventOptionalData(DailyEvent singleDailyEvent) {
        assertEquals(NOTES, singleDailyEvent.getNotes());
        assertEquals(TITLE, singleDailyEvent.getTitle());
    }

    private Date createPlusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.plusDays(1).toDate();
    }

    private Date createMinusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.minusDays(1).toDate();
    }

    @Test
    public void noDailyEventFoundForPatient() {
        saveFilledInDailyEvent();

        DailyEvents dailyEvents = dataLayer.getDailyEvents("1");
        assertEquals(0, dailyEvents.getSize());
    }

    @Test
    public void noDailyEventFoundForId() {
        saveFilledInDailyEvent();

        DailyEvent singleDailyEvent = dataLayer.getDailyEvent(null, "1");
        assertNull(singleDailyEvent);
    }

    @Test
    public void noDailyEventFoundBetweenDates() {
        saveFilledInDailyEvent();

        DateTime now = new DateTime(entryDate);
        Date tenDaysAgo = now.minusDays(10).toDate();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(tenDaysAgo), createPlusOne(tenDaysAgo));

        DailyEvents dailyEvents = dataLayer.getDailyEvents(patientId, filter);
        assertEquals(0, dailyEvents.getSize());
    }
}
