package com.agilex.healthcare.mobilehealthplatform.datalayer.document;

import java.io.*;
import java.net.*;
import java.util.*;

import javax.annotation.*;
import javax.imageio.*;
import javax.imageio.stream.*;

import org.apache.commons.io.*;
import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.datalayer.*;
import com.agilex.healthcare.mobilehealthplatform.domain.*;

public class DocumentDataLayerRepoTest extends SpringBasedIntegrationRepo {
    @Resource
    DocumentDataLayerRepo dataLayer;

    String userId;
    private final String STAR_WARS = "Star Wars";
    private final String STAR_WARS_NUM = "4";
    private final String EMPIRE_STRIKES_BACK = "The Empire Strikes Back";
    private final String EMPIRE_STRIKES_BACK_NUM = "5";
    private final String EMPIRE_STRIKES_BACK_NUM_A = "5A";
    private final String RETURN_OF_THE_JEDI = "Return of the Jedi";
    private final String RETURN_OF_THE_JEDI_NUM = "6";

    @Before
    public void before() {
        userId = UUID.randomUUID().toString();
    }

    @After
    public void after() {
        Documents documents = dataLayer.fetchDocuments(userId, null);
        for(Document document : documents) {
            dataLayer.deleteDocument(document);
        }
    }

    @Test
    public void verifySave() {
        Document document = createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);
        assertNotNull(document);
        assertEquals(STAR_WARS_NUM, document.getParentData().getUniqueId());
    }

    @Test
    public void verifyUpdate() {
        Document document = createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);

        assertEquals(STAR_WARS, document.getDocumentTitle());

        document.setDocumentTitle(EMPIRE_STRIKES_BACK);
        Document savedDocument = dataLayer.saveDocument(document);

        assertEquals(EMPIRE_STRIKES_BACK, savedDocument.getDocumentTitle());
    }

    @Test
    public void verifyGetSingle() {
        Document document = createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);
        Document fetchedDocument = dataLayer.fetchDocument(userId, document.getUniqueId());

        assertEquals(document.getDocumentTitle(), fetchedDocument.getDocumentTitle());
        assertEquals(document.getClassCode(), fetchedDocument.getClassCode());
        assertEquals(document.getId(), fetchedDocument.getId());
        assertEquals(document.getPatientId(), fetchedDocument.getPatientId());
        assertEquals(document.getBody().getMimeType(), fetchedDocument.getBody().getMimeType());
        assertEquals(document.getBody().getUri(), fetchedDocument.getBody().getUri());
        assertEquals(document.getParentData().getUniqueId(), fetchedDocument.getParentData().getUniqueId());

        assertEquals("JPEG", getImageMimeType(fetchedDocument.getBody().getContents()));
     }

    private String getImageMimeType(byte[] contents) {
        try {
            ByteArrayInputStream bis = new ByteArrayInputStream(contents);
            ImageInputStream picStream = ImageIO.createImageInputStream(bis);

            Iterator iter = ImageIO.getImageReaders(picStream);
            if (!iter.hasNext()) {
                return null;
            }

            ImageReader reader = (ImageReader)iter.next();

            picStream.close();

            return reader.getFormatName();
        } catch (IOException e) {
            fail(e.getMessage());
        }
        return null;
    }
    
    @Test
    public void verifyOnlySelf() {
    	Document document = createAndSaveDocument(RETURN_OF_THE_JEDI, RETURN_OF_THE_JEDI_NUM);
    	assertNotNull(document);
    	
    	document = dataLayer.fetchDocument(userId, document.getUniqueId());
    	assertNotNull(document);
    	
    	document = dataLayer.fetchDocument("other user", RETURN_OF_THE_JEDI_NUM);
    	assertNull(document);
    }

    @Test
    public void verifyFetchMultiple() {
        createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);
        createAndSaveDocument(EMPIRE_STRIKES_BACK, EMPIRE_STRIKES_BACK_NUM);

        Documents documents = dataLayer.fetchDocuments(userId, new DocumentFilter());

        assertNotNull(documents);
        assertEquals(2, documents.size());
    }

    @Test
    public void verifyFetchMultipleWithTitleFilter() {
        createAndSaveDocument(RETURN_OF_THE_JEDI, RETURN_OF_THE_JEDI_NUM);

        createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);
        createAndSaveDocument(EMPIRE_STRIKES_BACK, EMPIRE_STRIKES_BACK_NUM);
        createAndSaveDocument(EMPIRE_STRIKES_BACK, EMPIRE_STRIKES_BACK_NUM_A);

        DocumentFilter filter = new DocumentFilter(null, EMPIRE_STRIKES_BACK);
        Documents documents = dataLayer.fetchDocuments(userId, filter);

        assertEquals(2, documents.size());
        assertEquals(EMPIRE_STRIKES_BACK, documents.get(0).getDocumentTitle());
        assertEquals(EMPIRE_STRIKES_BACK, documents.get(1).getDocumentTitle());
    }

    @Test
    public void verifyFetchWithEventIdFilter() {
        createAndSaveDocument(RETURN_OF_THE_JEDI, RETURN_OF_THE_JEDI_NUM);

        createAndSaveDocument(STAR_WARS, STAR_WARS_NUM);
        createAndSaveDocument(EMPIRE_STRIKES_BACK, EMPIRE_STRIKES_BACK_NUM);
        createAndSaveDocument(EMPIRE_STRIKES_BACK, EMPIRE_STRIKES_BACK_NUM_A);

        RelatedDataItem relatedDataItem = new RelatedDataItem();
        relatedDataItem.setUniqueId(RETURN_OF_THE_JEDI_NUM);
        DocumentFilter filter = new DocumentFilter(relatedDataItem, null);

        Documents documents = dataLayer.fetchDocuments(userId, filter);
        assertEquals(1, documents.size());
        assertEquals(RETURN_OF_THE_JEDI, documents.get(0).getDocumentTitle());
        assertEquals(RETURN_OF_THE_JEDI_NUM, documents.get(0).getParentData().getUniqueId());
    }

    private Document createAndSaveDocument(String title, String entryId) {
        Document doc = new Document();
        doc.setClassCode("jedi");
        doc.setDocumentTitle(title);
        doc.setPatientId(userId);

        addBody(doc);
        addEntryId(entryId, doc);

        return dataLayer.saveDocument(doc);
    }

    private void addBody(Document doc) {
        doc.setMimeType("image/jpeg");
        addUri(doc);
        addContents(doc);
    }

    private void addUri(Document doc) {
        try {
            doc.getBody().setUri(new URI("in/a/galaxy/far/far/away"));
        } catch (URISyntaxException e) {
            fail(e.getMessage());
        }
    }

    private void addContents(Document doc) {
        InputStream picStream = DocumentDataLayerRepoTest.class.getResourceAsStream("football.jpg");
        try {
            doc.getBody().setContents(IOUtils.toByteArray(picStream));
            picStream.close();
        } catch (IOException e) {
            fail(e.getMessage());
        }
    }

    private void addEntryId(String entryId, Document doc) {
        if (entryId == null)
            return;

        RelatedDataItem relatedDataItem = new RelatedDataItem();
        relatedDataItem.setUniqueId(entryId);
        doc.setParentData(relatedDataItem);
    }
}
