package com.agilex.healthcare.mobilehealthplatform.datalayer.diet;

import java.util.*;

import javax.annotation.Resource;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import org.junit.*;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.utility.MealType;

public class DietDataLayerRepoTest extends UserHistoryIntegrationRepo {

	private static final String integrationTestPatientId = "integrationTestPatientId";
	
	@Resource
	DietDataLayerRepo dietDataLayer;
	
	@After 
	public void cleanup() {
		DietEntries entries = dietDataLayer.getDietEntries(integrationTestPatientId);
		for(DietEntry entry : entries) {
			dietDataLayer.deleteDietEntry(entry);
		}
	}
	
	@Test
	public void testGetAllDietEntries() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("1");
		entry.setProtein("1");
		entry.setCalories("1");
		entry.setFat("1");
		entry.setMealType(MealType.LUNCH);
	
		dietDataLayer.saveDietEntry(entry);
		
		entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes 2");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("2");
		entry.setProtein("2");
		entry.setCalories("2");
		entry.setFat("2");
		entry.setMealType(MealType.LUNCH);
		

		dietDataLayer.saveDietEntry(entry);

		DietEntries dietEntries = dietDataLayer.getDietEntries(integrationTestPatientId);
		assertNotNull(dietEntries);
		assertTrue(dietEntries.size() == 2);
	}

	@Test
	public void verifyRetrieveForSingleUser() {
		DietEntry savedDiet = createAndSaveDietEntry();
		assertNotNull(savedDiet);
		
		DietEntry dietEntry = dietDataLayer.getDietEntry(integrationTestPatientId, savedDiet.getUniqueId());
		assertNotNull(dietEntry);
		
		DietEntry shouldBeNull = dietDataLayer.getDietEntry("otherPatientId", savedDiet.getUniqueId());
		assertNull(shouldBeNull);
	}
	
	@Test
	public void testGetDietEntryByStartDateAndEndDate() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("1");
		entry.setProtein("1");
		entry.setCalories("1");
		entry.setFat("1");
		entry.setMealType(MealType.LUNCH);
		
		try {
			Thread.sleep(1000);
		} catch (InterruptedException occassionalPersistenceDelayCausesFailures) {
		}

		DietEntry dietEntry = dietDataLayer.saveDietEntry(entry);
		assertNotNull(dietEntry.getId());
        DateFilter filter = DateFilterFactory.createFilterFromDate(getYesterday(), new Date());
		DietEntries shouldBeOne = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter = DateFilterFactory.createFilterFromDate(new Date(), new Date());
		DietEntries shouldBeEmpty = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeEmpty.size());
	}
	
	@Test
	public void verifyUpdate() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("1");
		entry.setProtein("1");
		entry.setCalories("1");
		entry.setFat("1");
		entry.setMealType(MealType.LUNCH);
		
		entry = dietDataLayer.saveDietEntry(entry);
		
		String notes = entry.getNotes();
		assertNotNull(notes);

		entry.setNotes("new notes");
		
		entry = dietDataLayer.saveDietEntry(entry);
		
		assertNotNull(entry.getNotes());
		
		assertEquals("new notes", entry.getNotes());
	}
	
	@Test
	public void testGetDietEntryByEndDate() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("1");
		entry.setProtein("1");
		entry.setCalories("1");
		entry.setFat("1");
		entry.setMealType(MealType.LUNCH);

		
		dietDataLayer.saveDietEntry(entry);
		
		DateFilter filter = DateFilterFactory.createFilterFromDate(null, getTomorrow());
		DietEntries shouldBeOne = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter = DateFilterFactory.createFilterFromDate(null, getYesterday());
		DietEntries shouldBeZilch = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeZilch.size());
	}

	@Test
	public void testGetDietEntryByStartDate() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("1");
		entry.setProtein("1");
		entry.setCalories("1");
		entry.setFat("1");
		entry.setMealType(MealType.LUNCH);
		
		dietDataLayer.saveDietEntry(entry);

        DateFilter filter = DateFilterFactory.createFilterFromDate(getYesterday(), null);
		DietEntries shouldBeOne = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(1, shouldBeOne.size());
		
		filter = DateFilterFactory.createFilterFromDate(getTomorrow(), null);
		DietEntries shouldBeZilch = dietDataLayer.getDietEntries(integrationTestPatientId, filter);
		assertEquals(0, shouldBeZilch.size());
	}
	
	DietEntry createAndSaveDietEntry() {
		DietEntry entry = new DietEntry();
		entry.setEntryDate(new Date());
		entry.setNotes("itest notes 2");
		entry.setPatientId(integrationTestPatientId);
		entry.setCarbs("2");
		entry.setProtein("2");
		entry.setCalories("2");
		entry.setFat("2");
		entry.setMealType(MealType.LUNCH);
		
		return dietDataLayer.saveDietEntry(entry);
	}

	Date getYesterday() {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, -1);
		return cal.getTime();
	}
	
	Date getTomorrow() {
		Calendar cal = Calendar.getInstance();
		cal.add(Calendar.DATE, +1);
		return cal.getTime();
	}
}
