package com.agilex.healthcare.mobilehealthplatform.datalayer.contactlog;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.*;
import com.agilex.healthcare.utility.ModeHelper;
import org.joda.time.DateTime;
import org.junit.*;

import javax.annotation.Resource;
import java.util.*;

public class ContactLogDataLayerRepoTest extends UserHistoryIntegrationRepo {
    public static final String NOTES = "go to the dagobah system";
    public static final String TASK = "learn the force";
    public static final String ACTIVITY = "visit";
    public static final String CONTACT = "yoda";

    @Resource
    ContactLogDataLayerRepo dataLayer;

    ContactLog contactLog;
    String patientId;
    Date logTime;

    @Before
    public void before() {
        patientId = UUID.randomUUID().toString();
        logTime = new Date();

        createContactLog();
    }

    private void createContactLog() {
        contactLog = new ContactLog();
        contactLog.setPatientId(patientId);
        contactLog.setLogTime(logTime);
        contactLog.setNote(NOTES);
        contactLog.setActivity(ACTIVITY);
        contactLog.setTask(TASK);
        contactLog.setContact(CONTACT);
    }

    @After
    public void after() {
        dataLayer.deleteContactLog(contactLog);
    }

    @Test
    public void updateContactLog() {
        saveFilledInContactLog();
        saveEmptyContactLog();
    }

    @Test
    public void verifyOnlyBySpecificPatient() {
    	saveFilledInContactLog();
    	
    	ContactLog log = dataLayer.getContactLog(patientId, contactLog.getUniqueId());
    	assertNotNull(log);
    	
    	ContactLog shouldBeNull = dataLayer.getContactLog("someoneElse", contactLog.getUniqueId());
    	assertNull(shouldBeNull);
    }
    
    @Test
    public void findByPatientId() {
        saveFilledInContactLog();

        ContactLogs contactLogs = dataLayer.getContactLogs(patientId);
        assertEquals(1, contactLogs.getSize());
        verifyContactLogOptionalData(contactLogs.get(0));
    }

    @Test
    public void findBeforeToday() {
        saveFilledInContactLog();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(logTime), null);
        ContactLogs contactLogs = dataLayer.getContactLogs(patientId, filter);

        assertEquals(1, contactLogs.getSize());
        verifyContactLogOptionalData(contactLogs.get(0));
    }

    @Test
    public void findAfterToday() {
        saveFilledInContactLog();

        DateFilter filter = DateFilterFactory.createFilterFromDate(null, createPlusOne(logTime));
        ContactLogs contactLogs = dataLayer.getContactLogs(patientId, filter);

        assertEquals(1, contactLogs.getSize());
        verifyContactLogOptionalData(contactLogs.get(0));
    }

    @Test
    public void findBetweenDates() {
        saveFilledInContactLog();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(logTime), createPlusOne(logTime));
        ContactLogs contactLogs = dataLayer.getContactLogs(patientId, filter);

        assertEquals(1, contactLogs.getSize());
        verifyContactLogOptionalData(contactLogs.get(0));
    }

    private void saveEmptyContactLog() {
        contactLog.setLogTime(null);
        contactLog.setNote(null);
        contactLog.setActivity(null);
        contactLog.setTask(null);
        contactLog.setContact(null);
        saveFilledInContactLog();

        assertNull(contactLog.getLogTime());
        assertNull(contactLog.getNote());
        assertNull(contactLog.getActivity());
        assertNull(contactLog.getTask());
        assertNull(contactLog.getContact());
    }

    private void saveFilledInContactLog() {
        contactLog = dataLayer.saveContactLog(contactLog);
        assertNotNull(contactLog);
        assertEquals(patientId, contactLog.getPatientId());
    }

    private void verifyContactLogOptionalData(ContactLog singleContactLog) {
        assertEquals(NOTES, singleContactLog.getNote());
        assertEquals(ACTIVITY, singleContactLog.getActivity());
        assertEquals(TASK, singleContactLog.getTask());
        assertEquals(CONTACT, singleContactLog.getContact());
    }

    private Date createPlusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.plusDays(1).toDate();
    }

    private Date createMinusOne(Date dt) {
        DateTime dtOrg = new DateTime(dt);
        return dtOrg.minusDays(1).toDate();
    }

    @Test
    public void noContactLogFoundForPatient() {
        saveFilledInContactLog();

        ContactLogs contactLogs = dataLayer.getContactLogs("1");
        assertEquals(0, contactLogs.getSize());
    }

    @Test
    public void noContactLogFoundForId() {
        saveFilledInContactLog();

        ContactLog contactLog = dataLayer.getContactLog(null, "1");
        assertNull(contactLog);
    }

    @Test
    public void noContactLogFoundBetweenDates() {
        saveFilledInContactLog();

        DateTime now = new DateTime(logTime);
        Date tenDaysAgo = now.minusDays(10).toDate();

        DateFilter filter = DateFilterFactory.createFilterFromDate(createMinusOne(tenDaysAgo), createPlusOne(tenDaysAgo));

        ContactLogs contactLogs = dataLayer.getContactLogs(patientId, filter);
        assertEquals(0, contactLogs.getSize());
    }
}
