package com.agilex.healthcare.mobilehealthplatform.datalayer.assessmentresult;

import java.util.*;

import javax.annotation.Resource;

import com.agilex.healthcare.mobilehealthplatform.datalayer.UserHistoryIntegrationRepo;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.mobilehealthplatform.domain.code.AssessmentCode;
import com.agilex.healthcare.utility.DateHelper;

public class AssessmentResultDataLayerRepoTest extends UserHistoryIntegrationRepo {
	@Resource
	AssessmentResultDataLayerRepo datalayer;

	Date dateTaken = new Date();
	AssessmentResult assessmentResult;

	@Test
	public void verifySaveOfBasicAssessmentResult() {
		String userId = UUID.randomUUID().toString();
		assessmentResult = createAndSaveBasicAssessmentResult(userId, dateTaken, 10, false);

		assertNotNull(assessmentResult);
		assertEquals("value for unique title", assessmentResult.getUniqueTitle());

        datalayer.delete(assessmentResult);
    }
	
	@Test
	public void verifyGetAssessmentResultForSpecificPatient() {
		String userId = UUID.randomUUID().toString();
		assessmentResult = createAndSaveBasicAssessmentResult(userId, dateTaken, 10, false);
		
		AssessmentResult retrieved = datalayer.fetchAssessmentResult(new PatientIdentifier("EDIPI", userId), assessmentResult.getUniqueId());
		assertNotNull(retrieved);
		
		String maliciousUserId = UUID.randomUUID().toString();
		retrieved = datalayer.fetchAssessmentResult(new PatientIdentifier("EDIPI", maliciousUserId), assessmentResult.getUniqueId());
		assertNull(retrieved);
	}

	@Test
	public void verifySaveReportText() {
		String userId = UUID.randomUUID().toString();
		assessmentResult = createAndSaveBasicAssessmentResult(userId, dateTaken, 10, false);

		assertNotNull(assessmentResult);
		assertEquals("sample report", assessmentResult.getAssessmentResultReport().getProperties().getValueByPropertyName(AssessmentCode.REPORT_SUMMARY));

        datalayer.delete(assessmentResult);
    }

	@Test
	public void verifySaveOfDraft() {
		String userId = UUID.randomUUID().toString();
		assessmentResult = createAndSaveBasicAssessmentResult(userId, dateTaken, 10, false);
        AssessmentResult assessmentResult2 = createAndSaveBasicAssessmentResult(userId, dateTaken, 12, true);

		AssessmentResults savedAssessmentResults = datalayer.fetchAssessmentDraftsByPatient(new PatientIdentifier("default", userId));
		assertNotNull(savedAssessmentResults);
		assertEquals(1, savedAssessmentResults.size());

        datalayer.delete(assessmentResult);
        datalayer.delete(assessmentResult2);
    }

	@Test
	public void verifyFetchPreviouslyCompletedResult() {
		String userId = UUID.randomUUID().toString();
		Date oldestDate = DateHelper.getPreviousMonth();
		Date middleDate = DateHelper.getYesterday();
		Date newestDate = DateHelper.getToday();

		assessmentResult = createAndSaveBasicAssessmentResult(userId, oldestDate, 12, false);
        AssessmentResult assessmentResult2 = createAndSaveBasicAssessmentResult(userId, middleDate, 11, false);
        AssessmentResult assessmentResult3 = createAndSaveBasicAssessmentResult(userId, newestDate, 10, false);

		AssessmentResult mostRecentlyCompletedResult = datalayer.fetchMostRecentlyCompletedResult("testAssessmentId", new PatientIdentifier("default", userId));
		assertEquals(10, mostRecentlyCompletedResult.getScore());

        datalayer.delete(assessmentResult);
        datalayer.delete(assessmentResult2);
        datalayer.delete(assessmentResult3);
    }
	
	@Test
	public void verifyFetchWithMultipleUsersAndSameAssessment() {
		String userId = UUID.randomUUID().toString();
		Date oldestDate = DateHelper.getPreviousMonth();
		
		assessmentResult = createAndSaveBasicAssessmentResult(userId, oldestDate, 12, false);
		AssessmentResult firstResult = datalayer.fetchMostRecentlyCompletedResult("testAssessmentId", new PatientIdentifier("default", userId));
		
		assertEquals(firstResult.getPatientId(), userId);
		
		userId = UUID.randomUUID().toString();
		assessmentResult = createAndSaveBasicAssessmentResult(userId, oldestDate, 12, false);
		AssessmentResult secondResult = datalayer.fetchMostRecentlyCompletedResult("testAssessmentId", new PatientIdentifier("default", userId));
		
		assertEquals(secondResult.getPatientId(), userId);
		
		assertFalse(firstResult.getPatientId().equals(userId));
	}

	private AssessmentResult createAndSaveBasicAssessmentResult(String userId, Date dateTaken, int score, boolean inProgress) {
		AssessmentResult assessmentResult = new AssessmentResult();
		assessmentResult.setPatientId(userId);
		assessmentResult.setDateTaken(dateTaken);
		assessmentResult.setAssessmentId("testAssessmentId");
		assessmentResult.setUniqueTitle("value for unique title");
		assessmentResult.setVersion("value for version");
		assessmentResult.setAuthenticationStrategy("value for authentication strategy");
		assessmentResult.setNotes("value for notes");
		assessmentResult.setScore(score);
		assessmentResult.setScoringAlgorithm(AssessmentCode.SCORING_ALGORITHM_UNSCORED);
		assessmentResult.setInProgress(inProgress);
		assessmentResult.setResponses(new AssessmentResponses());

		AssessmentResultReport assessmentResultReport = new AssessmentResultReport();
		assessmentResultReport.getProperties().add(new AssessmentProperty(AssessmentCode.REPORT_SUMMARY, "sample report"));
		assessmentResult.setAssessmentResultReport(assessmentResultReport);

		assessmentResult.setWaitingPeriod(7);

        return datalayer.saveAssessmentResult(assessmentResult);
	}
}