package com.agilex.healthcare.mobilehealthplatform.datalayer.assessment;

import com.agilex.healthcare.mobilehealthplatform.datalayer.SpringBasedIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.healthcare.utility.ModeHelper;
import org.junit.*;

import javax.annotation.Resource;

public class AssessmentDataLayerRepoTest extends SpringBasedIntegrationRepo {

    private static final String A_NOTE = "active complete assessment";
    private static final String BEST_SCORE_ALG = "best";
    private static final String A_TITLE = "alive";
    private static final String VERSION_1_1 = "1.1";
    private static final String AUTH_STRATEGY = "random";
    private static final int WAITING_PERIOD = 5;

    @Resource
    AssessmentDataLayerRepo dataLayer;

    private Assessment healthAssessment;
    private Assessment activeAssessment;
    private Assessment incompleteAssessment;

    @BeforeClass
    public static void checkWhetherTestsAreRunningInVeteranDevMode() {
        Assume.assumeTrue(ModeHelper.isMode("va-veteran-dev"));
    }

    @Before
    public void setup() {
        healthAssessment = dataLayer.saveAssessment(createInHealthAssessmentWithoutQuestions());
        activeAssessment = dataLayer.saveAssessment(createCompleteActiveAssessment());
        incompleteAssessment = dataLayer.saveAssessment(createIncompleteAssessment());
    }

    private Assessment createInHealthAssessmentWithoutQuestions() {
        Assessment assessment = new Assessment();

        assessment.setScoringAlgorithm(BEST_SCORE_ALG);
        assessment.setUniqueTitle(A_TITLE);
        assessment.setVersion(VERSION_1_1);
        assessment.setWaitingPeriod(WAITING_PERIOD);
        assessment.setAuthenticationStrategy(AUTH_STRATEGY);
        assessment.setInHealthAssessment(true);
        assessment.setNotes("inactive health assessment");

        return assessment;
    }

    private Assessment createCompleteActiveAssessment() {
        Assessment assessment = new Assessment();

        assessment.setScoringAlgorithm(BEST_SCORE_ALG);
        assessment.setUniqueTitle(A_TITLE);
        assessment.setVersion(VERSION_1_1);
        assessment.setWaitingPeriod(WAITING_PERIOD);
        assessment.setAuthenticationStrategy(AUTH_STRATEGY);
        assessment.setPublish(true);
        assessment.setNotes(A_NOTE);

        AssessmentQuestion firstQuestion = createQuestion(1, "multiple choice", "arrow");
        firstQuestion.getQuestionProperties().add(new AssessmentProperty("connection", "venus"));
        firstQuestion.getQuestionProperties().add(new AssessmentProperty("timeout", "60000"));
        firstQuestion.getQuestionChoices().add(createChoice("1"));

        AssessmentQuestionChoice choice = createChoice("2");
        choice.setProperties(new AssessmentProperties());
        choice.getProperties().add(new AssessmentProperty("connection", "venus"));
        firstQuestion.getQuestionChoices().add(choice);

        assessment.getQuestions().add(firstQuestion);

        AssessmentQuestion secondQuestion = createQuestion(1, "multiple choice", "arrow");
        secondQuestion.getQuestionProperties().add(new AssessmentProperty("connection", "venus"));
        secondQuestion.getQuestionProperties().add(new AssessmentProperty("timeout", "60000"));
        secondQuestion.getQuestionChoices().add(createChoice("a"));

        AssessmentQuestionChoice secondChoice = createChoice("b");
        secondChoice.setProperties(new AssessmentProperties());
        secondChoice.getProperties().add(new AssessmentProperty("planet", "venus"));
        secondQuestion.getQuestionChoices().add(secondChoice);

        assessment.getQuestions().add(secondQuestion);

        return assessment;
    }

    private Assessment createIncompleteAssessment() {
        Assessment assessment = new Assessment();

        assessment.setScoringAlgorithm(BEST_SCORE_ALG);
        assessment.setUniqueTitle(A_TITLE);
        assessment.setVersion(VERSION_1_1);
        assessment.setWaitingPeriod(WAITING_PERIOD);
        assessment.setAuthenticationStrategy(AUTH_STRATEGY);
        assessment.setNotes("incomplete assessment");

        AssessmentQuestion firstQuestion = createQuestion(1, "multiple choice", "arrow");
        firstQuestion.getQuestionChoices().add(createChoice("1"));

        AssessmentQuestionChoice choice = createChoice("2");
        choice.setProperties(new AssessmentProperties());
        choice.getProperties().add(new AssessmentProperty("connection", "venus"));
        firstQuestion.getQuestionChoices().add(choice);

        assessment.getQuestions().add(firstQuestion);

        AssessmentQuestion secondQuestion = createQuestion(1, "multiple choice", "arrow");
        secondQuestion.getQuestionProperties().add(new AssessmentProperty("connection", "venus"));
        secondQuestion.getQuestionProperties().add(new AssessmentProperty("timeout", "60000"));

        assessment.getQuestions().add(secondQuestion);

        return assessment;
    }

    private AssessmentQuestion createQuestion(int number, String type, String prompt) {
        AssessmentQuestion question = new AssessmentQuestion();

        question.setPrompt(prompt);
        question.setType(type);
        question.setQuestionNumber(number);

        return question;
    }

    private AssessmentQuestionChoice createChoice(String number) {
        AssessmentQuestionChoice choice = new AssessmentQuestionChoice();

        choice.setChoiceNumber(number);

        return choice;
    }

    @After
    public void cleanup() {
        dataLayer.deleteAssessment(healthAssessment);
        dataLayer.deleteAssessment(activeAssessment);
        dataLayer.deleteAssessment(incompleteAssessment);
    }

    @Test
    public void fetchHealthAssessments() {
        Assessments assessments = dataLayer.fetchAssessmentsForHealthAssessment();

        Assessment assessment = assessments.get(0);
        assertEquals(healthAssessment.getId(), assessment.getId());
    }

    @Test
    public void fetchActiveAssessments() {
        Assessments assessments = dataLayer.fetchActiveAssessments();

        Assessment assessment = assessments.get(0);
        assertEquals(activeAssessment.getId(), assessment.getId());
    }

    @Test
    public void fetchAssessmentById() {
        Assessment assessment = dataLayer.fetchAssessment(activeAssessment.getId());

        assertEquals(BEST_SCORE_ALG, assessment.getScoringAlgorithm());
        assertEquals(A_TITLE, assessment.getUniqueTitle());
        assertEquals(VERSION_1_1, assessment.getVersion());
        assertEquals(WAITING_PERIOD, assessment.getWaitingPeriod());
        assertEquals(AUTH_STRATEGY, assessment.getAuthenticationStrategy());
        assertTrue(assessment.getPublish());
        assertFalse(assessment.isInHealthAssessment());
        assertEquals(A_NOTE, assessment.getNotes());
        assertEquals(2, assessment.getQuestions().size());
        assertEquals(2, assessment.getQuestions().get(0).getQuestionChoices().size());
        assertEquals(2, assessment.getQuestions().get(0).getQuestionProperties().size());
    }

    @Test
    public void fetchQuestionById() {
        AssessmentQuestion question = dataLayer.fetchQuestion(activeAssessment.getId(), activeAssessment.getQuestions().get(1).getId());

        assertEquals(2, question.getQuestionProperties().size());
        assertEquals(2, question.getQuestionChoices().size());
    }

    @Test
    public void updateAssessment() {
        incompleteAssessment.setPublish(true);

        dataLayer.saveAssessment(incompleteAssessment);
        Assessments assessments = dataLayer.fetchActiveAssessments();

        assertEquals(2, assessments.size());
    }
}
