package com.agilex.healthcare.mobilehealthplatform.datalayer.appointment;

import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import javax.annotation.Resource;
import javax.ws.rs.WebApplicationException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.SpringBasedIntegrationRepo;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequest;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessage;
import com.agilex.healthcare.mobilehealthplatform.domain.AppointmentRequestMessages;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.DetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestProviderOption;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestType;
import com.agilex.healthcare.mobilehealthplatform.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.utility.DateHelper;

/**
 * Created with IntelliJ IDEA.
 * User: clarkee
 * Date: 1/30/13
 * Time: 11:07 AM
 * To change this template use File | Settings | File Templates.
 */
public class AppointmentRequestMessageDataLayerRepoTest extends SpringBasedIntegrationRepo {

    @Resource
    AppointmentRequestDataLayerRepo datalayer;

    Date createdDate;
	String appointmentType;
	String visitType;
	String facilityName;
	String email;
	String phoneNumber;
	String status;
	String optionDate1;
	String optionTime1;
	String optionDate2;
	String optionTime2;
	String optionDate3;
	String optionTime3;
	String purposeOfVisit;
	String otherPurposeOfVisit;
	String providerId;
	String providerName;
	String providerPersonClass;
	String providerOption;
	boolean secondRequest;

	Set<String> bestTimetoCall;

	String testPatientId;


	private AppointmentRequest appointmentRequest1;
	
    @Before
    public void setupData() throws Exception {
        super.setUp();
        testPatientId = "D123401";
		appointmentRequest1 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123231234", new DOBDate(new Date()));
		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "1000";
		String facilityName = "Facility 1";
		String parentSiteCode = "test parent site code";
		String facilityType = "test facility type";
		Facility facility = createFacility(facilityCode, facilityName, facilityType, facilityAddress, facilityCity, facilityState, parentSiteCode);

		appointmentRequest1.setFacility(facility);
		appointmentRequest1.setStatus(AppointmentRequestStatus.CANCELLED.getName());
		addARDetailCode(appointmentRequest1, "DETCODE8");
		addARDetailCode(appointmentRequest1, "DETCODE7");
		appointmentRequest1 = datalayer.saveAppointmentRequest(appointmentRequest1);
    }

    @After
    public void removeData() throws Exception {
        super.tearDown();
        datalayer.deleteAppointmentRequest(appointmentRequest1);
    }

    private void addARDetailCode(AppointmentRequest appointmentRequest, String code) {

		DetailCode detailCode = new DetailCode();
		detailCode.setCode(code);

		List<AppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
		AppointmentRequestDetailCode arDetailCode = new AppointmentRequestDetailCode();
		arDetailCode.setCreatedDate(new Date());
		arDetailCode.setDetailCode(detailCode);
		arDetailCode.setUserId("john.smith");
		arDetailCodes.add(arDetailCode);
		appointmentRequest.setAppointmentRequestDetailCode(arDetailCodes);
	}
    
    protected void init() {
		appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
		visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
		facilityName = "Test Facility Name";
		email = "test@agilex.com";
		phoneNumber = "555-555-5555";
		Calendar calendar = Calendar.getInstance();
		createdDate = calendar.getTime();
		calendar.add(Calendar.DATE, 3);
		optionDate1 = DateHelper.formatDate(calendar.getTime());
		optionTime1 = "AM";
		calendar.add(Calendar.DATE, 1);
		optionDate2 = DateHelper.formatDate(calendar.getTime());
		optionTime2 = "PM";
		calendar.add(Calendar.DATE, 2);
		optionDate3 = DateHelper.formatDate(calendar.getTime());
		optionTime3 = "AM";
		status = AppointmentRequestStatus.SUBMITTED.getName();
		purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
		otherPurposeOfVisit = "Other purpose of visit custom";
		providerId = "PROV1";
		providerName = "John      ";
		providerPersonClass = "Physician";
		providerOption = AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName();
		secondRequest = true;

		bestTimetoCall = new HashSet<String>();
		bestTimetoCall.add("9 AM - 11 AM");
		bestTimetoCall.add("11 AM - 1 PM");
	}
    
	protected AppointmentRequest createBasicAppointmentRequest(String patientId, String firstName, String lastName, String ssn, DOBDate dateOfBirth) {
		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "537";
		String parentSiteCode = "test parent site code";
		String facilityType = "test facility type";
		Facility facility = createFacility(facilityCode, facilityName, facilityType, facilityAddress, facilityCity, facilityState, parentSiteCode);

		AppointmentRequest appointmentRequest = new AppointmentRequest();
		
		appointmentRequest.setCreatedDate(createdDate);
		appointmentRequest.setAppointmentType(appointmentType);
		appointmentRequest.setVisitType(visitType);
		appointmentRequest.setFacility(facility);
		appointmentRequest.setEmail(email);
		appointmentRequest.setPhoneNumber(phoneNumber);
		appointmentRequest.setStatus(status);
		appointmentRequest.setOptionDate1(optionDate1);
		appointmentRequest.setOptionTime1(optionTime1);
		appointmentRequest.setOptionDate2(optionDate2);
		appointmentRequest.setOptionTime2(optionTime2);
		appointmentRequest.setOptionDate3(optionDate3);
		appointmentRequest.setOptionTime3(optionTime3);
		appointmentRequest.setBestTimetoCall(bestTimetoCall);
		appointmentRequest.setPurposeOfVisit(purposeOfVisit);
		appointmentRequest.setOtherPurposeOfVisit(otherPurposeOfVisit);
		appointmentRequest.setProviderId(providerId);
		appointmentRequest.setProviderName(providerName);
		appointmentRequest.setProviderPersonClass(providerPersonClass);
		
		appointmentRequest.setTextMessagingAllowed(false);
		appointmentRequest.setTextMessagingPhoneNumber("555-555-5555");

		providerOption = getRandomProviderOption();
		appointmentRequest.setProviderOption(providerOption);

		appointmentRequest.setLastUpdatedDate(new Date());
		
		appointmentRequest.setPatient(createPatient(patientId, firstName, lastName, ssn, dateOfBirth));
		appointmentRequest.setSecondRequest(secondRequest);

		return appointmentRequest;
	}

	protected Patient createPatient(String patientId, String firstName, String lastName, String ssn, DOBDate dateOfBirth) {
		Patient patient = new Patient();
		patient.setId(patientId);
		patient.setFirstName(firstName);
		patient.setLastName(lastName);
		patient.setSsn(ssn);
		patient.setDateOfBirth(dateOfBirth);
		
		return patient;
	}
	
	protected Facility createFacility(String facilityCode, String facilityName, String facilityType, String facilityAddress, String facilityCity, String facilityState, String parentSiteCode) {
		Facility facility = new Facility();
		facility.setAddress(facilityAddress);
		facility.setCity(facilityCity);
		facility.setFacilityCode(facilityCode);
		facility.setName(facilityName);
		facility.setParentSiteCode(parentSiteCode);
		facility.setState(facilityState);
		facility.setType(facilityType);

		return facility;
	}
	
	protected String getRandomProviderOption() {
		String result = "";
		Random random = new Random();
		if (random.nextInt(9) >= 5) {
			result = AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName();
		}

		return result;
	}
	
    //Basic save test, not verifying, only ensuring operation completes
    @Test
    public void saveAppointmentRequestMessage(){
        AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequest1.getUniqueId());
        message.setMessageDateTime(DateHelper.getToday());
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setSenderId("StaffId");
        message.setMessageText("Please call ASAP");
        boolean isProvider = false;
        
        //Make sure you can't fetch by a different patient Id
        AppointmentRequestMessage result = null;
        try {
        	result = datalayer.saveAppointmentRequestMessage("fake patient Id should result in failure", message, isProvider);
        } catch (WebApplicationException e) {
        }
        assertNull(result);
        
        //Make sure the data id was populated
        result = datalayer.saveAppointmentRequestMessage(message.getPatientId(), message,isProvider);
        assertNotNull(result);
        assertNotNull(result.getDataIdentifier());
        String uniqueId = result.getDataIdentifier().getUniqueId();
        assertNotNull(uniqueId);
        assertFalse(uniqueId.equals(""));
    }

    @Test
    public void getAppointmentRequestMessage(){
        final AppointmentRequestMessages messages = datalayer.fetchAppointmentRequestMessages("None", "noMessagesId");
        assertNotNull(messages);
        assertEquals(0, messages.size());
    }
    //Basic save test, not verifying, only ensuring operation completes
    @Test
    public void saveAndRetrieveAppointmentRequestMessage(){
        String appointmentRequestId = appointmentRequest1.getUniqueId();
        //save initial message
        AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequestId);
        message.setMessageDateTime(DateHelper.getToday());
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setSenderId("StaffId");
        message.setMessageText("Please call ASAP");
        boolean isProvider = false;
        AppointmentRequestMessage result = datalayer.saveAppointmentRequestMessage(message.getPatientId(), message,isProvider);

        assertNotNull(result);

        final AppointmentRequestMessages messages = datalayer.fetchAppointmentRequestMessages(message.getPatientId(), appointmentRequestId);
        assertNotNull(messages);
        assertEquals(1, messages.size());
    }
    @Test
    public void updateVeteranMessageFlagTest(){
    	AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequest1.getUniqueId());
        message.setMessageDateTime(DateHelper.getToday());
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setSenderId("StaffId");
        message.setMessageText("Please call ASAP");
        boolean isProvider = true;
        AppointmentRequestMessage result = datalayer.saveAppointmentRequestMessage(message.getPatientId(), message,isProvider);
        assertNotNull(result);
        
        //Verify that hasVeteranNewMessage flag is updated to true.
        AppointmentRequest ar = datalayer.fetchAppointmentRequest(message.getPatientId(), appointmentRequest1.getUniqueId());
        assertEquals(true, ar.isHasVeteranNewMessage());
        
        //Try to mark as veteran messages read with different id.
        AppointmentRequest updatedAR = null;
        try {
        	updatedAR = datalayer.updateMessageFlag("fake id, should not work", appointmentRequest1.getUniqueId(), false);
        } catch (WebApplicationException e) {
        }
        assertNull(updatedAR);
        
        //Mark as Veteran read all new messages.
        updatedAR = datalayer.updateMessageFlag(message.getPatientId(), appointmentRequest1.getUniqueId(), false);
        assertEquals(false, updatedAR.isHasVeteranNewMessage());
    }
    @Test
    public void updateProviderMessageFlagTest(){
    	AppointmentRequestMessage message = new AppointmentRequestMessage();
        message.setAppointmentRequestId(appointmentRequest1.getUniqueId());
        message.setMessageDateTime(DateHelper.getToday());
        message.setPatientIdentifier("EDIPI", "D123401");
        message.setSenderId("StaffId");
        message.setMessageText("Please call ASAP");
        boolean isProvider = false;  // Veteran posts new message
        AppointmentRequestMessage result = datalayer.saveAppointmentRequestMessage(message.getPatientId(), message,isProvider);
        assertNotNull(result);
        
        //Verify that hasProviderNewMessage flag is updated to true.
        AppointmentRequest ar = datalayer.fetchAppointmentRequest(message.getPatientId(), appointmentRequest1.getUniqueId());
        assertEquals(true, ar.isHasProviderNewMessage());
        

        //Try to mark as provider messages read with different veteran id.
        AppointmentRequest updatedAR = null;
        try {
        	updatedAR = datalayer.updateMessageFlag("fake id, should not work", appointmentRequest1.getUniqueId(), true);
        } catch (WebApplicationException e) {
        }
        assertNull(updatedAR);
        
        
        //Mark as Provider read all new messages.
        updatedAR = datalayer.updateMessageFlag(message.getPatientId(), appointmentRequest1.getUniqueId(), true);
        assertEquals(false, updatedAR.isHasProviderNewMessage());
    }
}
