package com.agilex.healthcare.utility.objectmapper;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import org.junit.Before;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntries;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalEntry;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalObservation;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalTableData;
import com.agilex.healthcare.mobilehealthplatform.domain.VitalTableRow;
import com.agilex.healthcare.mobilehealthplatform.domain.code.ObservationTypeCode;
import com.agilex.healthcare.mobilehealthplatform.domain.code.VitalSectionCode;
import com.agilex.healthcare.utility.DateHelper;


public class VitalTableDataTest {

	private VitalTableDataGenerator generator;

	@Before
	public void setUp() {
		generator = new VitalTableDataGenerator();
	} 

	@Test
	public void validateNullEntryReturnsEmptyTable(){
		VitalTableData vitalTableData = generator.generateVitalTableData(null);
		assertNotNull(vitalTableData);
	}
	
	@Test
	public void validateEmptyEntryReturnsEmptyTable(){
		VitalEntries entries = new VitalEntries();
		VitalTableData vitalTableData = generator.generateVitalTableData(entries);
		assertNotNull(vitalTableData);
	}
	

	@Test
	public void validateSimpleConversion(){
		
		VitalEntries vitalEntries = createVitalEntriesWithOneVitalEntry();
		
		VitalTableData vitalTableData = generator.generateVitalTableData(vitalEntries);
		assertEquals(1, vitalTableData.size());
		assertEquals("130", vitalTableData.get(0).getSystolic());
		assertEquals("80", vitalTableData.get(0).getDiastolic());
		assertEquals("60", vitalTableData.get(0).getPulse());
	}

	
	
	@Test
	public void validateConversionWithMultipleEntries(){
		
		VitalEntries vitalEntries = createSampleData();
		
		VitalTableData vitalTableData = generator.generateVitalTableData(vitalEntries);
		assertEquals(3, vitalTableData.size());

		for (VitalTableRow vitalTableRow : vitalTableData) {
			String date = DateHelper.format(vitalTableRow.getEntryTime(), "MM/dd/yyyy");
			if("12/01/2011".equalsIgnoreCase(date)){
				assertEquals("120", vitalTableRow.getSystolic());
				assertEquals("65", vitalTableRow.getDiastolic());
				assertEquals("62", vitalTableRow.getPulse());
			}else if("12/02/2011".equalsIgnoreCase(date)){
				assertEquals("135", vitalTableRow.getSystolic());
				assertEquals("75", vitalTableRow.getDiastolic());
				assertEquals("55", vitalTableRow.getPulse());
				assertNull(vitalTableRow.getRespiration());
			}else if("12/03/2011".equalsIgnoreCase(date)){
				assertEquals("136", vitalTableRow.getSystolic());
				assertEquals("76", vitalTableRow.getDiastolic());
				assertEquals("56", vitalTableRow.getPulse());
				assertEquals("95", vitalTableRow.getTemperature());
				assertEquals("30", vitalTableRow.getRespiration());
				assertEquals("150", vitalTableRow.getWeight());
				assertEquals("5", vitalTableRow.getPain());
			}

		}
	}
	
	@Test
	public void validateSortingInDescendingOrder(){
		VitalEntries vitalEntries = createSampleData();
		VitalTableData vitalTableData = generator.generateVitalTableData(vitalEntries);
		vitalTableData.sortDescending("entryTime");
		assertEquals("12/03/2011", DateHelper.format(vitalTableData.get(0).getEntryTime(), "MM/dd/yyyy"));
		assertEquals("12/02/2011", DateHelper.format(vitalTableData.get(1).getEntryTime(), "MM/dd/yyyy"));
		assertEquals("12/01/2011", DateHelper.format(vitalTableData.get(2).getEntryTime(), "MM/dd/yyyy"));
	}

	@Test
	public void validateSortingInAscendingOrder(){
		VitalEntries vitalEntries = createSampleData();
		VitalTableData vitalTableData = generator.generateVitalTableData(vitalEntries);
		vitalTableData.sortAscending("entryTime");
		assertEquals("12/01/2011", DateHelper.format(vitalTableData.get(0).getEntryTime(), "MM/dd/yyyy"));
		assertEquals("12/02/2011", DateHelper.format(vitalTableData.get(1).getEntryTime(), "MM/dd/yyyy"));
		assertEquals("12/03/2011", DateHelper.format(vitalTableData.get(2).getEntryTime(), "MM/dd/yyyy"));
	}

	@Test(expected=Exception.class)
	public void verifyExceptionisThrownWhileSortingWhenPropertyIsNotSet(){
		VitalEntries vitalEntries = createSampleData();
		VitalTableData vitalTableData = generator.generateVitalTableData(vitalEntries);
		vitalTableData.sortAscending(null);
	}

	
	private VitalEntries createVitalEntriesWithOneVitalEntry() {
		VitalEntries vitalEntries = new VitalEntries();
		String aldie     PatientId = "PATID26";

		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/1/2011 10:00:00"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		VitalObservation observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("60");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("130");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("80");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEIGHT);
		observation.setValue("65");
		observation.setValueUnits("inches");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);
		
		return vitalEntries;
	}
	
	
	private VitalEntries createSampleData() {
		VitalEntries vitalEntries = new VitalEntries();
		String aldie     PatientId = "PATID26";

		VitalEntry vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDateTime("12/1/2011 10:00:00"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		VitalObservation observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("60");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("80");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("130");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDateTime("12/1/2011 11:35:00"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("62");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("120");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("65");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDateTime("12/1/2011 06:35:00"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("95");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("129");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("69");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);
		
		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/2/2011"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("55");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("135");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("75");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);
		
		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.BloodPressure);
		vitalEntry.setId("A");
		vitalEntry.setEntryDate(DateHelper.parseDate("12/3/2011"));
		vitalEntry.setQualifier("Standing");
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEART_BEAT); // pulse
		observation.setValue("56");
		observation.setValueUnits("beats");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_SYSTOLIC); // systolic
		observation.setValue("136");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.INTRAVASCULAR_DIASTOLIC); // diasystolic
		observation.setValue("76");
		observation.setValueUnits("mm(hg)");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntry.setNotes("test note");
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.Temperature);
		vitalEntry.setEntryDate(DateHelper.parseDate("12/3/2011"));
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.BODY_TEMPERATURE);
		observation.setValue("95");
		observation.setValueUnits("some units");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.Respiration);
		vitalEntry.setEntryDate(DateHelper.parseDate("12/3/2011"));
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.RESPIRATION);
		observation.setValue("30");
		observation.setValueUnits("some units");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.Weight);
		vitalEntry.setEntryDate(DateHelper.parseDate("12/3/2011"));
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.HEIGHT);
		observation.setValue("65");
		observation.setValueUnits("inches");
		vitalEntry.getVitalObservations().add(observation);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.WEIGHT);
		observation.setValue("150");
		observation.setValueUnits("some units");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntries.add(vitalEntry);

		vitalEntry = new VitalEntry();
		vitalEntry.setSection(VitalSectionCode.Pain);
		vitalEntry.setEntryDate(DateHelper.parseDate("12/3/2011"));
		vitalEntry.setPatientId(aldie     PatientId);
		observation = new VitalObservation();
		observation.setType(ObservationTypeCode.PAIN);
		observation.setValue("5");
		observation.setValueUnits("some units");
		vitalEntry.getVitalObservations().add(observation);
		vitalEntries.add(vitalEntry);

		
		return vitalEntries;
	}


}
