package com.agilex.healthcare.mobilehealthplatfrom.domain.filter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import junit.framework.Assert;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.medication.MedicationFilterer;
import com.agilex.healthcare.mobilehealthplatform.domain.Medication;
import com.agilex.healthcare.mobilehealthplatform.domain.Medications;
import com.agilex.healthcare.utility.DateHelper;

public class MedicationFiltererTest {

    @Test
	public void verifyHandleZeroMedications() {
		Medications medications = new Medications();
		MedicationFilter filter = new MedicationFilter();
		Medications filteredMedications = MedicationFilterer.filter(medications, filter);

		assertEquals(0, filteredMedications.size());
	}

	@Test
	public void verifySourceFilter() {
		MedicationFilter filter = new MedicationFilter();
		filter.setSourceSystems(addValueToList("VA"));
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
		assertEquals("VA", filteredMedications.get(0).getSourceSystem());
	}

	@Test
	public void verifyDateFilter() {
		MedicationFilter filter = new MedicationFilter(null, DateHelper.parseDate("12/1/2011"), DateHelper.get30DaysFromNow(), null, null, null);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
	}

	@Test
	public void verifySingleStatusFilter() {
		List<String> statuses = addValueToList("Active");
		MedicationFilter filter = new MedicationFilter(statuses, null, null, null, null, null);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
		assertEquals(1, filter.getStatuses().size());
		assertTrue("Active".equalsIgnoreCase(filteredMedications.get(0).getStatus()));
	}

	@Test
	public void verifyMultipleStatusFilter() {
		List<String> statuses = addValueToList("Active", "Expired");
		MedicationFilter filter = new MedicationFilter(statuses, null, null, null, null, null);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(2, filteredMedications.size());
	}

	@Test
	public void verifySingleSourceSystemFilter() {
		List<String> sourceSystems = addValueToList("VA");
		MedicationFilter filter = new MedicationFilter(null, null, null, null, null, sourceSystems);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
	}

	@Test
	public void verifyMultipleSourceSystemFilter() {
		List<String> sourceSystems = addValueToList("VA", "NonVA");
		MedicationFilter filter = new MedicationFilter(null, null, null, null, null, sourceSystems);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(2, filteredMedications.size());
	}

	@Test
	public void verifySingleSourceSystem() {
		List<String> sourceSystems = addValueToList("VA");
		MedicationFilter filter = new MedicationFilter();
		filter.setSourceSystems(sourceSystems);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
	}

	@Test
	public void verify90DayFilterWithExpirationDates() {
		Medication medication1 = new Medication();
		medication1.setDrugName("medication1");
		medication1.setStatus("ACTIVE");
		medication1.setExpiredDate(DateHelper.get30DaysFromNow());
		
		Medication medication2 = new Medication();
		medication2.setDrugName("medication2");
		medication2.setStatus("EXPIRED");
		medication2.setExpiredDate(DateHelper.minusDays(new Date(), 91));
		
		MedicationFilter filter = new MedicationFilter();
		List<String> statuses = addValueToList("Active");
		filter.setStatuses(statuses);
		Medications medications = new Medications(medication1, medication2);
		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		
		Assert.assertEquals(1, filteredMedications.size());
	}
	
	@Test
	public void verify90DayFilterWithoutExpirationDates() {
		Medication medication1 = new Medication();
		medication1.setDrugName("medication1");
		medication1.setStatus("ACTIVE");
		
		Medication medication2 = new Medication();
		medication2.setDrugName("medication2");
		medication2.setStatus("EXPIRED");
		
		MedicationFilter filter = new MedicationFilter();
		List<String> statuses = addValueToList("Active");
		filter.setStatuses(statuses);
		Medications medications = new Medications(medication1, medication2);
		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		
		Assert.assertEquals(1, filteredMedications.size());
	}
	
	@Test
	public void verifySingleMedicationSourceFilter() {
		List<String> medicationSources = addValueToList("Inpatient");
		MedicationFilter filter = new MedicationFilter(null, null, null, null, medicationSources, null);
		Medications medications = createMedications();

		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
	}

	@Test
	public void verifyMultipleMedicationSourceFilter() {
		List<String> medicationSources = addValueToList("Inpatient", "Outpatient");
		MedicationFilter filter = new MedicationFilter(null, null, null, null, medicationSources, null);
		
		Medications medications = new Medications();
		
		Medication medication = new Medication();
		medication.setDrugName("medication1");
		medication.setMedicationSource("Inpatient");
		medication.setExpiredDate(DateHelper.getToday());
		medication.setStartDate(DateHelper.minusDays(new Date(), 365));
		medication.setStatus("Expired");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("medication2");
		medication.setMedicationSource("Outpatient");
		medication.setExpiredDate(DateHelper.getToday());
		medication.setStartDate(DateHelper.minusDays(new Date(), 365));
		medication.setStatus("active");
		medications.add(medication);
		
		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(2, filteredMedications.size());
	}

	@Test
	public void verifyPagination() {
		MedicationFilter filter = new MedicationFilter();
		filter.setPage("1");
		filter.setResultCount("1");

		Medications medications = createMedications();
		Medications filteredMedications = MedicationFilterer.filter(medications, filter);
		assertEquals(1, filteredMedications.size());
	}

	@Test
	public void verifyPaginationThrowErrorWithMissingData() {
		MedicationFilter filter = new MedicationFilter();
		filter.setPage("1");

		Medications medications = createMedications();
		try {
			MedicationFilterer.filter(medications, filter);
			assertTrue(false);
		} catch (UnsupportedOperationException shouldFail) {
			assertTrue(true);
		}
	}
	
	@Test
	public void verifyActiveFilteringWithTranslatedStatus() {
		MedicationFilter filter = new MedicationFilter();
		filter.addStatus("active");
		Medications medications = createMedicationsThatNeedTranslation();
		
		Medications filtered = MedicationFilterer.filter(medications, filter);
		assertEquals(2, filtered.size());
	}

	@Test
	public void verifyExpiredFilteringWithTranslatedStatus() {
		MedicationFilter filter = new MedicationFilter();
		filter.addStatus("expired");
		Medications medications = createMedicationsThatNeedTranslation();
		
		Medications filtered = MedicationFilterer.filter(medications, filter);
		assertEquals(7, filtered.size());
	}

	@Test
	public void verifyNonVASourceSystemWithTranslatedStatus() {
		MedicationFilter filter = new MedicationFilter();
		filter.addSourceSystem("Non-VA");
		Medications medications = createMedicationsThatNeedTranslation();
		
		Medications filtered = MedicationFilterer.filter(medications, filter);
		assertEquals(2, filtered.size());
	}

	@Test
	public void verifyVASourceSystemWithTranslatedStatus() {
		MedicationFilter filter = new MedicationFilter();
		filter.addSourceSystem("VA");
		Medications medications = createMedicationsThatNeedTranslation();
		
		Medications filtered = MedicationFilterer.filter(medications, filter);
		assertEquals(7, filtered.size());
	}

	private Medication createMedicationMatch() {
		Medication medication = new Medication();
		medication.setDrugName("VA Inpatient Active [today]");
		medication.setSourceSystem("VA");
		medication.setMedicationSource("Inpatient");
		medication.setStatus("Active");
		
		medication.setStartDate(DateHelper.getToday());
		medication.setExpiredDate(DateHelper.get30DaysFromNow());

		return medication;
	}

	private Medication createMedicationMismatch() {
		Medication medication = new Medication();
		medication.setDrugName("other outpatient expired 1/1/2020");
		medication.setSourceSystem("Other");
		medication.setMedicationSource("Outpatient");
		medication.setStatus("Expired");
		medication.setStartDate(DateHelper.parseDate("1/1/2020"));

		return medication;
	}

	private Medications createMedications() {
		Medications medications = new Medications();
		Medication med1 = createMedicationMatch();
		Medication med2 = createMedicationMismatch();
		medications.add(med1);
		medications.add(med2);

		return medications;
	}
	
	private Medications createMedicationsThatNeedTranslation() {
        Date expiredDate;
        Date nonExpiredDate;

        expiredDate = determineValidExpiredMedicationDate();
        nonExpiredDate = determineValidNonExpiredMedicationDate();

        Medications medications = new Medications();

        Medication medication = new Medication();
        medication.setDrugName("VA Inpatient Active 1/1/2011");
        medication.setStartDate(DateHelper.parseDate("1/01/2011"));
        medication.setExpiredDate(expiredDate);
        medication.setSourceSystem("VA");
        medication.setMedicationSource("INPATIENT");
        medication.setStatus("ACTIVE");
        medications.add(medication);

        medication = new Medication();
        medication.setDrugName("VA Inpatient Refill 1/1/2011");
        medication.setStartDate(DateHelper.parseDate("1/01/2011"));
        medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("REFILL");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("VA Inpatient expired 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
        medication.setExpiredDate(expiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("EXPIRED");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("VA Inpatient discontinued 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("DISCONTINUED");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("VA Inpatient suspended 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("SUSPENDED");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("VA Inpatient deleted 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("DELETED");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("VA Inpatient done 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("DONE");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("Other Inpatient disc. by provider 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2011"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("Other");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("DISCONTINUED BY PROVIDER");
		medications.add(medication);
		
		medication = new Medication();
		medication.setDrugName("non-VA Inpatient disc. (edit) 1/1/2011");
		medication.setStartDate(DateHelper.parseDate("1/01/2012"));
		medication.setExpiredDate(nonExpiredDate);
		medication.setSourceSystem("Non-VA");
		medication.setMedicationSource("INPATIENT");
		medication.setStatus("DISCONTINUED (EDIT)");
		medications.add(medication);
		
		return medications;
	}

    private Date determineValidExpiredMedicationDate() {
        //needs to return date more than 90 days from current date
        return DateHelper.minusDays(DateHelper.getToday(), 91);
    }

    private Date determineValidNonExpiredMedicationDate() {
        //needs to return date within 90 days of current date
        return  DateHelper.minusDays(DateHelper.getToday(), 90);
    }

    private List<String> addValueToList(String... values) {
		List<String> criteria = new ArrayList<String>();
		for (int i = 0; i < values.length; i++) {
			criteria.add(values[i]);
		}
		return criteria;
	}
}
