package com.agilex.healthcare.mobilehealthplatfrom.domain.filter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import java.util.UUID;

import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentFilter;
import com.agilex.healthcare.mobilehealthplatform.datalayer.document.DocumentFilterer;
import com.agilex.healthcare.mobilehealthplatform.domain.Document;
import com.agilex.healthcare.mobilehealthplatform.domain.Documents;
import com.agilex.healthcare.mobilehealthplatform.domain.RelatedDataItem;
import com.agilex.healthcare.mobilehealthplatform.domain.code.EventCode;

public class DocumentFiltererTest {

	@Test
	public void testSimpleFilter() {
		String communityId = UUID.randomUUID().toString();
		String uniqueId = UUID.randomUUID().toString();
		RelatedDataItem parentItem = initializeRelatedData(EventCode.DIET, communityId, uniqueId);
		Documents documents = createTwoSimpleDocuments(communityId, uniqueId, "Title Test");
		DocumentFilter filter = new DocumentFilter(parentItem, "Title Test");
		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void testWithNullFilter() {
		Documents documents = createTwoSimpleDocuments("dontcare", "dontcare", "NullFilter");
		Documents filteredDocuments = DocumentFilterer.filter(documents, null);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void filterNullDocumentsList() {
		Documents filteredDocuments = DocumentFilterer.filter(null, new DocumentFilter());
		assertNull(filteredDocuments);
	}

	@Test
	public void filterWithNullFilter() {
		Documents documents = createTwoSimpleDocuments("dontcare", "dontcare", "dontcare");
		Documents filteredDocuments = DocumentFilterer.filter(documents, null);
		assertEquals(2, filteredDocuments.size());
	}

//	@Test
//	public void verifyFilterOutDomain() {
//		Documents documents = createTwoSimpleDocuments("dontcare", "dontcare", "dontcare");
//		RelatedDataItem filterItem = new RelatedDataItem("fail", null, null);
//		RelatedDataItem documentItem = initializeRelatedData(EventCode.DIET, null, null);
//	}

	@Test
	public void testWithoutDocumentTitle() {
		RelatedDataItem parentItem = initializeRelatedData(EventCode.DIET, "docTitleTest", "docTitleTest");
		Documents documents = createTwoSimpleDocuments("docTitleTest", "docTitleTest", "docTitleTest");
		DocumentFilter filter = new DocumentFilter(parentItem, null);

		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void testWithoutUniqueId() {
		RelatedDataItem parentItem = initializeRelatedData(EventCode.DIET, "noUnique", null);
		Documents documents = createTwoSimpleDocuments("noUnique", "noUnique", "noUnique");
		DocumentFilter filter = new DocumentFilter(parentItem, "noUnique");

		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void testWithoutCommunity() {
		RelatedDataItem parentItem = initializeRelatedData(EventCode.DIET, null, "noCommunity");
		Documents documents = createTwoSimpleDocuments("noCommunity", "noCommunity", "noCommunity");
		DocumentFilter filter = new DocumentFilter(parentItem, "noCommunity");

		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void testNullRelatedDataOnDocument() {
		Document document = new Document();
		document.setDocumentTitle("nullParentData");

		Documents documents = appendToDocuments(document);

		assertEquals(1, documents.size());
	}

	@Test
	public void testWithoutDomain() {
		RelatedDataItem parentItem = initializeRelatedData(null, "noDomain", "noDomain");
		Documents documents = createTwoSimpleDocuments("noDomain", "noDomain", "noDomain");
		DocumentFilter filter = new DocumentFilter(parentItem, "noDomain");

		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(2, filteredDocuments.size());
	}

	@Test
	public void verifyFilterWithDomainMismatch() {
		RelatedDataItem parentItem = initializeRelatedData(EventCode.DAILY, "community", "unique");
		Documents documents = createTwoSimpleDocuments("community", "unique", "title");
		DocumentFilter filter = new DocumentFilter(parentItem, "title");

		Documents filteredDocuments = DocumentFilterer.filter(documents, filter);

		assertEquals(0, filteredDocuments.size());
	}

	private Documents createTwoSimpleDocuments(String communityId, String uniqueId, String title) {
		String patientId = UUID.randomUUID().toString();

		RelatedDataItem parentItem = initializeRelatedData(EventCode.DIET, communityId, uniqueId);
		Document doc1 = createSimpleDocument(patientId, title, parentItem);
		Document doc2 = createSimpleDocument(patientId, title, parentItem);
		Documents documents = appendToDocuments(doc1, doc2);

		return documents;
	}

	private Documents appendToDocuments(Document... documents) {
		Documents appendedDocuments = new Documents();
		for (Document document : documents) {
			appendedDocuments.add(document);
		}
		return appendedDocuments;
	}

	private Document createSimpleDocument(String patientId, String documentTitle, RelatedDataItem parentItem) {
		Document document = new Document();
		document.setPatientId(patientId);
		document.setDocumentTitle(documentTitle);
		document.setParentData(parentItem);

		return document;
	}

	public DocumentFilter initializeFilter(RelatedDataItem parentItem, String documentTitle) {
		return new DocumentFilter(parentItem, documentTitle);
	}

	public RelatedDataItem initializeRelatedData(String domain, String communityId, String uniqueId) {
		RelatedDataItem related = new RelatedDataItem();
		related.setDomain(domain);
		related.setCommunityId(communityId);
		related.setUniqueId(uniqueId);

		return related;
	}
}
